
package org.interactivemesh.jfx.sample3d.tuxcube;

import java.text.NumberFormat;

import javafx.animation.AnimationTimer;

import javafx.application.Application;

import javafx.beans.value.ChangeListener;
import javafx.beans.value.ObservableValue;

import javafx.collections.FXCollections;
import javafx.collections.ObservableList;

import javafx.event.ActionEvent;
import javafx.event.EventHandler;

import javafx.geometry.HPos;
import javafx.geometry.Rectangle2D;
import javafx.geometry.VPos;

import javafx.scene.Group;
import javafx.scene.Scene;
import javafx.scene.SceneAntialiasing;
import javafx.scene.SubScene;

import javafx.scene.control.CheckBox;
import javafx.scene.control.ComboBox;
import javafx.scene.control.ContextMenu;
import javafx.scene.control.Label;
import javafx.scene.control.ListCell;
import javafx.scene.control.ListView;
import javafx.scene.control.Menu;
import javafx.scene.control.MenuButton;
import javafx.scene.control.MenuItem;
import javafx.scene.control.RadioMenuItem;
import javafx.scene.control.SeparatorMenuItem;
import javafx.scene.control.Slider;
import javafx.scene.control.ToggleGroup;
import javafx.scene.control.Tooltip;

import javafx.scene.input.MouseButton;
import javafx.scene.input.MouseEvent;

import javafx.scene.layout.Background;
import javafx.scene.layout.BackgroundFill;
import javafx.scene.layout.ColumnConstraints;
import javafx.scene.layout.GridPane;
import javafx.scene.layout.Pane;
import javafx.scene.layout.RowConstraints;

import javafx.scene.paint.Color;
import javafx.scene.paint.CycleMethod;
import javafx.scene.paint.LinearGradient;
import javafx.scene.paint.RadialGradient;
import javafx.scene.paint.Stop;

import javafx.scene.shape.DrawMode;
import javafx.scene.shape.Rectangle;

import javafx.scene.text.Font;
import javafx.scene.text.FontWeight;

import javafx.stage.Screen;
import javafx.stage.Stage;
import javafx.stage.WindowEvent;

import javafx.util.Callback;

import org.interactivemesh.jfx.sample3d.tuxcube.FXTuxCubeSubScene.VP;

//import com.sun.javafx.perf.PerformanceTracker;

/**
 * org.interactivemesh.jfx.sample3d.tuxcube.FXTuxCube.java
 *
 * Version: 0.7.1
 * Date: 2013/10/31
 * 
 * Author:
 * August Lammersdorf, InteractiveMesh e.K.
 * Hauptstraße 28d, 85737 Ismaning
 * Germany / Munich Area
 * www.InteractiveMesh.com/org
 *
 * Please create your own implementation.
 * This source code is provided "AS IS", without warranty of any kind.
 * You are allowed to copy and use all lines you like of this source code
 * without any copyright notice,
 * but you may not modify, compile, or distribute this 'FXTuxCube.java'.
 *
 */
public final class FXTuxCube extends Application {

    /**
     * The main() method is ignored in correctly deployed JavaFX application.
     * main() serves only as fallback in case the application can not be
     * launched through deployment artifacts, e.g., in IDEs with limited FX
     * support. NetBeans ignores main().
     *
     * @param args the command line arguments
     */
    public static void main(String[] args) {
        launch(args);
    }
    
    private SubScene subScene = null;
    
    private long lastTime = 0L;
    private int frameCounter = 0;
    private int elapsedFrames = 100;
    
    private AnimationTimer fpsTimer = null;
    private HUDLabel fpsTitleLabel = null;
    private HUDLabel fpsLabel = null;
    private HUDLabel mpfTitleLabel = null;
    private HUDLabel mpfLabel = null;
    
    private boolean isUpdateFPS = false;
    private boolean isTuxRotating = false;
    private boolean isCubeRotating = false;          
    private boolean isMouseDragged = false;
    
    private DrawMode drawMode = DrawMode.FILL;
    
    private Background blackBG = null;
    private Background blueBG = null;
    private Background greenBG = null;
    
    private int gap = 0;
    private int border = 0;
    
    private Font titleFont = null;
    private Font textFont = null;
    private Font cellFont = null;
    
    private NumberFormat numFormat = null;    
    
    @Override
    public void start(Stage stage) {
                       
        final Rectangle2D screenRect = Screen.getPrimary().getBounds();
        final double screenWidth = screenRect.getWidth();
        final double screenHeight = screenRect.getHeight();

        // screenHeight > 1200      (1440/27', 1600/30')
        int cellFontSize = 16;
        int textFontSize = 20;
        int titleFontSize = 38;
        gap = 6;
        border = 50;

        // 900 < screenHeight  <= 1080/23'
        if (screenHeight <= 1080) {
            cellFontSize = 14;
            textFontSize = 16;
            titleFontSize = 30;
            gap = 4;
            border = 30;
        }
        // 1080 < screenHeight <= 1200/24'
        else if (screenHeight <= 1200) {
            cellFontSize = 14;
            textFontSize = 18;
            titleFontSize = 34;
            gap = 5;
            border = 40;
        }
        
        final String fontFamily = "Dialog";

        titleFont = Font.font(fontFamily, FontWeight.NORMAL, titleFontSize);
        textFont = Font.font(fontFamily, FontWeight.NORMAL, textFontSize);
        cellFont = Font.font(fontFamily, FontWeight.NORMAL, cellFontSize);

        numFormat = NumberFormat.getIntegerInstance();
        numFormat.setGroupingUsed(true);
                
        //
        // 3D subscene
        //
        final FXTuxCubeSubScene tuxCubeSubScene = new FXTuxCubeSubScene(); 
        
        subScene = tuxCubeSubScene.getSubScene();
        if (subScene == null) {
            exit();
        }
         
        //
        // Title
        //
        
        final HUDLabel titleLeftLabel = new HUDLabel("FXTuxCube", titleFont);       
        final HUDLabel titleRightLabel = new HUDLabel("JavaFX 8 3D", titleFont);
        
        //
        // 3D scene details and performance
        //
        
        // FPS - frames per second
           
        fpsTitleLabel = new HUDLabel("F P S");
        fpsTitleLabel.setTooltip(new Tooltip("frames per second"));
        fpsLabel = new HUDLabel("0");
        fpsLabel.setTooltip(new Tooltip("frames per second"));
        
        // MPF - milliseconds per frame
           
        mpfTitleLabel = new HUDLabel("M P F");
        mpfTitleLabel.setTooltip(new Tooltip("milliseconds per frame"));
        mpfLabel = new HUDLabel("0");
        mpfLabel.setTooltip(new Tooltip("milliseconds per frame"));
                
        // Tuxes/Shape3Ds/triangles
        
        final HUDLabel tuxesLabel = new HUDLabel("Tuxes");
        tuxesLabel.setTooltip(new Tooltip("number of Tux models and RotateTransitions"));
        final HUDLabel shape3dLabel = new HUDLabel("Shape3Ds");
        shape3dLabel.setTooltip(new Tooltip("number of Shape3D nodes"));
        final HUDLabel triangleLabel = new HUDLabel("Triangles");
        triangleLabel.setTooltip(new Tooltip("number of triangles"));
        
        // Set initial output values for autosizing
        final HUDLabel numTuxesLabel = new HUDLabel(numFormat.format(27));
        numTuxesLabel.setTooltip(new Tooltip("number of Tux models and RotateTransitions"));
        final HUDLabel numShape3dLabel = new HUDLabel(numFormat.format(162));
        numShape3dLabel.setTooltip(new Tooltip("number of Shape3D nodes"));
        final HUDLabel numTriaLabel = new HUDLabel(numFormat.format(371088));
        numTriaLabel.setTooltip(new Tooltip("number of triangles"));
       
        // Size of FXCanvas3D
        
        final HUDLabel heightLabel = new HUDLabel("Height");
        heightLabel.setTooltip(new Tooltip("height of 3D SubScene"));
        final HUDLabel pixHeightLabel = new HUDLabel("0");        
        pixHeightLabel.setTooltip(new Tooltip("height of 3D SubScene"));
        final HUDLabel widthLabel = new HUDLabel("Width");
        widthLabel.setTooltip(new Tooltip("width of 3D SubScene"));
        final HUDLabel pixWidthLabel = new HUDLabel("0");        
        pixWidthLabel.setTooltip(new Tooltip("width of 3D SubScene"));

        // Collect all outputs 
        
        final Rectangle gap1 = new Rectangle(gap, gap, Color.TRANSPARENT);
        final Rectangle gap2 = new Rectangle(gap, gap, Color.TRANSPARENT);
        
        final GridPane outputPane = new GridPane();
        outputPane.setHgap(10);
        outputPane.setVgap(0);
        outputPane.setGridLinesVisible(false);
        
        outputPane.add(fpsLabel, 0, 0);
        outputPane.add(fpsTitleLabel, 1, 0);
        outputPane.add(mpfLabel, 0, 1);
        outputPane.add(mpfTitleLabel, 1, 1);       
        outputPane.add(gap1, 0, 2);       
        outputPane.add(numTuxesLabel, 0, 3);
        outputPane.add(tuxesLabel, 1, 3);
        outputPane.add(numShape3dLabel, 0, 4);
        outputPane.add(shape3dLabel, 1, 4);
        outputPane.add(numTriaLabel, 0, 5);
        outputPane.add(triangleLabel, 1, 5);
        outputPane.add(gap2, 0, 6);
        outputPane.add(pixWidthLabel, 0, 7);
        outputPane.add(widthLabel, 1, 7);
        outputPane.add(pixHeightLabel, 0, 8);
        outputPane.add(heightLabel, 1, 8);
        
        final ColumnConstraints leftColumn = new ColumnConstraints();
        leftColumn.setHalignment(HPos.RIGHT);
        
        final ColumnConstraints rightColumn = new ColumnConstraints();
        rightColumn.setHalignment(HPos.LEFT);
       
        outputPane.getColumnConstraints().addAll(leftColumn, rightColumn);
        
        //
        // Controls
        //
        
        // Cube size
        
        final HUDLabel numTitleLabel = new HUDLabel("Cube");
        numTitleLabel.setTooltip(new Tooltip("width x height x depth"));
        
        final ObservableList<Number> nums = FXCollections.<Number>observableArrayList(1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12);
        
        final ComboBox<Number> cubeCombo = new ComboBox<>();
        cubeCombo.setTooltip(new Tooltip("width x height x depth"));
        cubeCombo.setItems(nums);
        cubeCombo.setVisibleRowCount(12);
        cubeCombo.getSelectionModel().select(2);               
        cubeCombo.getSelectionModel().selectedItemProperty().addListener(
            new ChangeListener<Number>() {
                @Override 
                public void changed(ObservableValue<? extends Number> ov, Number old_val, Number new_val) {  
                    int num = (Integer)new_val;
                    int num3 = num*num*num;
                    
                    tuxCubeSubScene.createTuxCubeOfDim(num, isTuxRotating, drawMode);      
                    tuxCubeSubScene.setVantagePoint(VP.FRONT);
                    
                    numTuxesLabel.setText( numFormat.format(num3) );
                    numShape3dLabel.setText( numFormat.format(num3*6) );
                    numTriaLabel.setText( numFormat.format(num3*13744) );
                }
            }
        );                
        cubeCombo.setButtonCell(new ListCell<Number>() {
            {
                this.setFont(cellFont);
            }      
            @Override protected void updateItem(Number item, boolean empty) {
                // calling super here is very important - don't skip this!
                super.updateItem(item, empty);                                               
                if (item != null) {
                    this.setText(Integer.toString((Integer)item));
                }
            }
        });       
        cubeCombo.setCellFactory(new Callback<ListView<Number>, ListCell<Number>>() {
            @Override public ListCell<Number> call(ListView<Number> p) {
                return new ListCell<Number>() {
                    {
                        this.setFont(cellFont);
                    }       
                    @Override protected void updateItem(Number item, boolean empty) {
                        // calling super here is very important - don't skip this!
                        super.updateItem(item, empty);                                               
                        if (item != null) {
                            this.setText(Integer.toString((Integer)item));
                        }
                    }
                };
            }
        });
        
        // Viewpoints   
        
        final HUDLabel vpTitleLabel = new HUDLabel("Viewpoint");
        vpTitleLabel.setTooltip(new Tooltip("select viewpoint"));
        
        // Prompt text workaround !!!!!!!!!!!!!!!!!!
        final ComboBox<VP> vpCombo = new ComboBox<VP>();
        vpCombo.setTooltip(new Tooltip("select viewpoint"));
        vpCombo.getItems().addAll(VP.BOTTOM, VP.CORNER, VP.FRONT, VP.TOP); // DO NOT add VP.Select !!
        // Pre-select the prompt text item
        vpCombo.setValue(VP.Select);       
        vpCombo.valueProperty().addListener(new ChangeListener<VP>() {
            @Override public void changed(ObservableValue<? extends VP> ov, VP old_val, VP new_val) {
                // Ignore the prompt text item VP.Select
                if (new_val != null && new_val != VP.Select) {
                    tuxCubeSubScene.setVantagePoint(new_val);
                    // Select the prompt text item
                    vpCombo.setValue(VP.Select);
                }
            }
        });            
        vpCombo.setButtonCell(new ListCell<VP>() {
            {
                this.setFont(cellFont);
            }
            @Override protected void updateItem(VP item, boolean empty) {
                // calling super here is very important - don't skip this!
                super.updateItem(item, empty);                                    
                if (item != null) {
                    this.setText(item.getListName());
                }
            }
        });       
        vpCombo.setCellFactory(new Callback<ListView<VP>, ListCell<VP>>() {
            @Override public ListCell<VP> call(ListView<VP> p) {
                return new ListCell<VP>() {
                    {
                        this.setFont(cellFont);
                    }
                    @Override protected void updateItem(VP item, boolean empty) {
                        // calling super here is very important - don't skip this!
                        super.updateItem(item, empty);
                        if (item != null) {
                            this.setText(item.getListName());
                        }
                    }
                };
            }
        });
       
        // Tux rotation
        
        final HUDLabel tuxRotTitlelabel = new HUDLabel("Tux");
        tuxRotTitlelabel.setTooltip(new Tooltip("start/pause rotation of Tuxes"));
       
        final CheckBox tuxRotCheck = new CheckBox();
        tuxRotCheck.setTooltip(new Tooltip("start/pause rotation of Tuxes"));
        tuxRotCheck.setStyle("-fx-label-padding: 0");
        tuxRotCheck.setGraphicTextGap(0);
        tuxRotCheck.setFont(textFont); // determines size of graphic !?
        tuxRotCheck.selectedProperty().addListener(new ChangeListener<Boolean>() {
            @Override
            public void changed(ObservableValue<? extends Boolean> ov, Boolean old_val, Boolean new_val) {
                isTuxRotating = !isTuxRotating;
                tuxCubeSubScene.playPauseTuxRotation(isTuxRotating);
                
                checkFPS();
            }
        });

        // Cube rotation
        
        final HUDLabel rotationLabel = new HUDLabel("    <  Cube Rotation  >    ");
        rotationLabel.setTooltip(new Tooltip("direction & speed of rotation"));
        
        final Slider rotationSlider = new Slider(20, 80, 50);
        rotationSlider.setBlockIncrement(0.6);
        rotationSlider.valueProperty().addListener(new ChangeListener<Number>(){
            @Override
            public void changed(ObservableValue<? extends Number> observable, Number oldValue, Number newValue) {
                isCubeRotating = (newValue.floatValue() < 49f || newValue.floatValue() > 51f);
                tuxCubeSubScene.setRotationSpeed(((Double)newValue).floatValue());
                
                checkFPS();
            }
        });
        rotationSlider.setTooltip(new Tooltip("direction & speed of rotation"));
      
        // Collect all controls
        
        final GridPane controlPane = new GridPane();
        controlPane.setHgap(10);
        controlPane.setVgap(4);
        controlPane.setGridLinesVisible(false);
        
        controlPane.add(numTitleLabel, 0, 0);
        controlPane.add(cubeCombo, 0, 1);
 
        controlPane.add(vpTitleLabel, 1, 0);
        controlPane.add(vpCombo, 1, 1);
        
        controlPane.add(tuxRotTitlelabel, 2, 0);
        controlPane.add(tuxRotCheck, 2, 1);
        
        controlPane.add(rotationLabel, 3, 0);
        controlPane.add(rotationSlider, 3, 1);     
        
        for (int i=0; i < 4; i++) {
            final ColumnConstraints cC = new ColumnConstraints();
            cC.setHalignment(HPos.CENTER);
            controlPane.getColumnConstraints().add(cC);
        }
       
        final RowConstraints topRow = new RowConstraints();
        topRow.setValignment(VPos.CENTER);
        
        final RowConstraints botRow = new RowConstraints();
        botRow.setValignment(VPos.CENTER);
       
        controlPane.getRowConstraints().addAll(topRow, botRow);


        final EventHandler onMouseEnteredHandler = new EventHandler<MouseEvent>() {
            @Override public void handle(MouseEvent e) {
                controlPane.setOpacity(1.0);
            }
        };
        final EventHandler onMouseExitedHandler = new EventHandler<MouseEvent>() {
            @Override public void handle(MouseEvent e) {
                controlPane.setOpacity(0.5);
            }
        };
                
        controlPane.setOpacity(0.5);
        controlPane.setOnMouseEntered(onMouseEnteredHandler);    
        controlPane.setOnMouseExited(onMouseExitedHandler);
       
        // Layout      
        
        // best smooth resizing if 3D-subScene is re-sized in Scene's ChangeListener
        // and HUDs in layeredPane.layoutChildren()
        
        // best background painting and mouse event handling
        // if subScene.setFill(Color.TRANSPARENT)
        // and layeredPane.setBackground(...). Don't use Scene.setFill.
        
                
        final double size = Math.min(screenWidth*0.8, screenHeight*0.8);
        subScene.setWidth(size);
        subScene.setHeight(size);
        
        final Group rootGroup = new Group();
        final Scene scene = new Scene(rootGroup, size, size, true);               
        final ChangeListener sceneBoundsListener = new ChangeListener() {
            @Override
            public void changed(ObservableValue observable, Object oldXY, Object newXY) {
                subScene.setWidth(scene.getWidth());
                subScene.setHeight(scene.getHeight());                            
             }
        };        
        scene.widthProperty().addListener(sceneBoundsListener);
        scene.heightProperty().addListener(sceneBoundsListener);
        
        final Pane layeredPane = new Pane() {
            @Override protected void layoutChildren() {
                
                double width = scene.getWidth();
                double height = scene.getHeight();

                titleLeftLabel.autosize();
                titleLeftLabel.relocate(border, border);
                
                titleRightLabel.autosize();
                titleRightLabel.relocate(width-titleRightLabel.getWidth()-border, border);
                
                controlPane.autosize();  
                controlPane.relocate(border, height - controlPane.getHeight() - border);  
                
                outputPane.autosize();   
                outputPane.relocate(width - border - outputPane.getWidth(), 
                                    height - outputPane.getHeight() - border);
                
                pixWidthLabel.setText(numFormat.format((int)width));
                pixHeightLabel.setText(numFormat.format((int)height));
            }
        };
        layeredPane.getChildren().addAll(subScene, titleLeftLabel, titleRightLabel, controlPane, outputPane);
         
        // Backgrounds 
        
        final Stop[] stopsRG = new Stop[]{new Stop(0.0, Color.LIGHTGRAY),
                                          new Stop(0.2, Color.BLACK),
                                          new Stop(1.0, Color.BLACK)};
        final RadialGradient rg = new RadialGradient(0, 0, 0.5, 0.5, 1, true, CycleMethod.NO_CYCLE, stopsRG);
        blackBG = new Background(new BackgroundFill(rg, null, null));
               
        final Stop[] stopsLG = new Stop[] {new Stop(0.0, Color.rgb(0, 73, 255)),
                                           new Stop(0.7, Color.rgb(127, 164, 255)), 
                                           new Stop(1.0, Color.rgb(0, 73, 255))};
        final LinearGradient lg = new LinearGradient(0, 0, 0, 1, true, CycleMethod.NO_CYCLE, stopsLG);   
        blueBG = new Background(new BackgroundFill(lg, null, null));
        
        greenBG = new Background(new BackgroundFill(Color.TURQUOISE, null, null));

        layeredPane.setBackground(blueBG); // initial background
        
        rootGroup.getChildren().add(layeredPane);
        
        //
        // ContextMenu 
        //
        
        // Projection
        final Menu menuProjection = new Menu("Projection");       
        final ToggleGroup toggleProjectionGroup = new ToggleGroup();
        
        final RadioMenuItem itemParallel = new RadioMenuItem("Parallel");
        itemParallel.setOnAction(new EventHandler<ActionEvent>() {
            @Override public void handle(ActionEvent e) {
                tuxCubeSubScene.setProjectionMode("Parallel");
            }
        });
        itemParallel.setToggleGroup(toggleProjectionGroup);
        itemParallel.setDisable(true); // Not implemented yet
        
        final RadioMenuItem itemPerspective = new RadioMenuItem("Perspective");
        itemPerspective.setOnAction(new EventHandler<ActionEvent>() {
            @Override public void handle(ActionEvent e) {
                tuxCubeSubScene.setProjectionMode("Perspective");
            }
        });
        itemPerspective.setSelected(true);
        itemPerspective.setToggleGroup(toggleProjectionGroup);
        
        menuProjection.getItems().addAll(itemParallel, itemPerspective);
        
        // Polygon mode
        final Menu menuPolygon = new Menu("Polygon mode");
        final ToggleGroup togglePolygonGroup = new ToggleGroup();
        
        final EventHandler polyModeHandler = new EventHandler<ActionEvent>() {
            @Override public void handle(ActionEvent e) {
                if (drawMode == DrawMode.FILL) {
                    drawMode = DrawMode.LINE;
                }
                else {
                    drawMode = DrawMode.FILL;
                }
                tuxCubeSubScene.setDrawMode(drawMode);
            }
        };
        
        final RadioMenuItem itemPolyFill = new RadioMenuItem("Fill");
        itemPolyFill.setToggleGroup(togglePolygonGroup);
        itemPolyFill.setSelected(true);
        itemPolyFill.setOnAction(polyModeHandler);
        
        final RadioMenuItem itemPolyLine = new RadioMenuItem("Line");
        itemPolyLine.setToggleGroup(togglePolygonGroup);
        itemPolyLine.setOnAction(polyModeHandler);
        
        menuPolygon.getItems().addAll(itemPolyFill, itemPolyLine);
        
        // Background
        final Menu menuBackground = new Menu("Background");       
        final ToggleGroup toggleBackgroundGroup = new ToggleGroup();
        
        final RadioMenuItem itemBlackBG = new RadioMenuItem("Black RadialGradient");
        itemBlackBG.setOnAction(new EventHandler<ActionEvent>() {
            @Override public void handle(ActionEvent e) {
                layeredPane.setBackground(blackBG);
            }
        });
        itemBlackBG.setToggleGroup(toggleBackgroundGroup);
        
        final RadioMenuItem itemBlueBG = new RadioMenuItem("Blue LinearGradient");
        itemBlueBG.setOnAction(new EventHandler<ActionEvent>() {
            @Override public void handle(ActionEvent e) {
                layeredPane.setBackground(blueBG);
            }
        });
        itemBlueBG.setToggleGroup(toggleBackgroundGroup);
        itemBlueBG.setSelected(true);

        final RadioMenuItem itemGreenBG = new RadioMenuItem("Turquois Color");
        itemGreenBG.setOnAction(new EventHandler<ActionEvent>() {
            @Override public void handle(ActionEvent e) {
                layeredPane.setBackground(greenBG);
            }
        });
        itemGreenBG.setToggleGroup(toggleBackgroundGroup);

        menuBackground.getItems().addAll(itemBlackBG, itemBlueBG, itemGreenBG);
 
        // Reset cube rotation
        final MenuItem itemStopCubeRot = new MenuItem("Reset cube rotation");
        itemStopCubeRot.setOnAction(new EventHandler<ActionEvent>() {
            @Override public void handle(ActionEvent e) {
                tuxCubeSubScene.stopCubeRotation();
                rotationSlider.setValue(50);
            }
        }); 
        
        // Reset tux rotation
        final MenuItem itemStopTuxRot = new MenuItem("Reset Tux rotation");
        itemStopTuxRot.setOnAction(new EventHandler<ActionEvent>() {
            @Override public void handle(ActionEvent e) {
                if (isTuxRotating) {
                    tuxRotCheck.setSelected(false);
                }
                tuxCubeSubScene.stopTuxRotation();
            }
        }); 
       
        // Scene anti-aliasing mode
        final Menu menuSceneAA = new Menu("Scene anti-aliasing");
        final ToggleGroup toggleGroupSceneAA = new ToggleGroup();
        
        final RadioMenuItem itemBalancedAA = new RadioMenuItem("BALANCED");
        itemBalancedAA.setToggleGroup(toggleGroupSceneAA);
        itemBalancedAA.setSelected(true);
        
        final RadioMenuItem itemDisabledAA = new RadioMenuItem("DISABLED");
        itemDisabledAA.setToggleGroup(toggleGroupSceneAA);
        
        final EventHandler sceneAAHandler = new EventHandler<ActionEvent>() {
            @Override public void handle(ActionEvent e) {
                
                SceneAntialiasing sceneAA = SceneAntialiasing.DISABLED;
                if (toggleGroupSceneAA.getSelectedToggle() == itemBalancedAA) {
                    sceneAA = SceneAntialiasing.BALANCED;
                }
                
                // Exchange SubScene

                subScene = tuxCubeSubScene.exchangeSubScene(sceneAA);
                layeredPane.getChildren().set(0, subScene);
            }
        };
        itemBalancedAA.setOnAction(sceneAAHandler);
        itemDisabledAA.setOnAction(sceneAAHandler);
        
        menuSceneAA.getItems().addAll(itemBalancedAA, itemDisabledAA);
        
        // Exit
        final MenuItem itemExit = new MenuItem("Exit");
        itemExit.setOnAction(new EventHandler<ActionEvent>() {
            @Override public void handle(ActionEvent e) {
                exit();
            }
        }); 
                
        final ContextMenu contextMenu = new ContextMenu();
        contextMenu.setHideOnEscape(true);       
        contextMenu.getItems().addAll(menuBackground, menuPolygon, menuProjection,
                                      new SeparatorMenuItem(), 
                                      itemStopCubeRot, itemStopTuxRot, 
                                      new SeparatorMenuItem(),
                                      menuSceneAA,
                                      new SeparatorMenuItem(), 
                                      itemExit);   
        
        layeredPane.setOnMouseClicked(new EventHandler<MouseEvent>() {
            @Override public void handle(MouseEvent e) {
                if (contextMenu.isShowing()) {
                    contextMenu.hide();
                }
                if (e.getButton() == MouseButton.SECONDARY && !isMouseDragged) {
                    contextMenu.show(layeredPane, e.getScreenX()+2, e.getScreenY()+2);
                }
                
                isMouseDragged = false;
               
                checkFPS();
            }
        });
        layeredPane.setOnDragDetected(new EventHandler<MouseEvent>() {
            @Override public void handle(MouseEvent e) { 
                isMouseDragged = true;
                
                checkFPS();
            }
        });         
        
        //
        // Stage
        //
        
        stage.setOnCloseRequest(new EventHandler<WindowEvent>() {
            @Override public void handle(WindowEvent event) {
                exit();
            }
        });
        stage.setScene(scene);    
        stage.setTitle("InteractiveMesh : FXTuxCube 0.7.1");
       
        stage.show();

        // Initial states
        
        lastTime = System.nanoTime();
        tuxCubeSubScene.createTuxCubeOfDim(3, isTuxRotating, drawMode);
        tuxCubeSubScene.setVantagePoint(VP.FRONT);

        //
        fpsTimer = new AnimationTimer() {
            @Override public void handle(long now) {
                
                if (isUpdateFPS == false) {
                    return;
                }

                frameCounter++;
                
                if (frameCounter > elapsedFrames) {

                    final long currTime = System.nanoTime();
                    final double duration = ((double)(currTime - lastTime)) / frameCounter;
                    lastTime = currTime;

                    // frames per second
                    final int fps = (int)(1000000000d / duration + 0.5);
                    // milliseconds per frame
                    final int mpf = (int)(duration/1000000d + 0.5);
                   
                    frameCounter = 0;
                    elapsedFrames = (int)Math.max(1, (fps/3f)); // update: 3 times per sec
                    elapsedFrames = Math.min(100, elapsedFrames); 

                    fpsLabel.setText(Integer.toString(fps));
                    mpfLabel.setText(Integer.toString(mpf));
                }
                /*
                float fpsf = PerformanceTracker.getSceneTracker(scene).getInstantFPS();
                System.out.println("fps     = " + fpsf + 
                                 "\npulses  = " + PerformanceTracker.getSceneTracker(scene).getInstantPulses() +
                                 "\nfps avg = " + PerformanceTracker.getSceneTracker(scene).getAverageFPS());
                */
            }
        };                
    }
    
    private void checkFPS() {
        boolean isRendering = isTuxRotating || isCubeRotating || isMouseDragged;
        if (isUpdateFPS != isRendering) {
            isUpdateFPS = isRendering;
        }
        else {
            return;
        }
        
        if (isUpdateFPS == false) {
            fpsTimer.stop();
            fpsLabel.setText(Integer.toString(0));
            mpfLabel.setText(Integer.toString(0));
        }
        else {
            fpsTimer.start(); 
        }
    }
   
    private void exit() {
        System.exit(0);    
    }
    
    // HUD : head-up display
    private final class HUDLabel extends Label {
        private HUDLabel(String text) {
            this(text, textFont);
        }
        private HUDLabel(String text, Font font) {
            super(text);
            setFont(font);
            setTextFill(Color.WHITE);            
        }
    }    
    
}
