package com.interactivemesh.j3d.sourcecode.navigation;

import javax.media.j3d.Behavior;
import javax.media.j3d.BoundingSphere;
import javax.media.j3d.Bounds;
import javax.media.j3d.BranchGroup;
import javax.media.j3d.Transform3D;
import javax.media.j3d.TransformGroup;
import javax.media.j3d.View;
import javax.media.j3d.ViewPlatform;
import javax.media.j3d.WakeupOnTransformChange;

import javax.vecmath.Point3d;
import javax.vecmath.Vector3d;

/**
 * ViewClipDistancer.
 *
 * This Behavior updates a View's back and front clip distances 
 * while the ViewPlatform's distance to the scene is changing.
 * If this scene is zoomed (by an OrbitBehavior(Interim)) 
 * the front and back clip distances are updated immediately.
 * Depending on the scene bounds these distances are calculated to achieve 
 * minimal clipping extent and optimized back/front ratio (< 1000).
 * 
 * Requires front and back clip policies as:
 * view.setBackClipPolicy(View.VIRTUAL_EYE);
 * view.setFrontClipPolicy(View.VIRTUAL_EYE);
 *
 * @author Copyright (c) 2007 August Lammersdorf, www.InteractiveMesh.com
 * @since October 06, 2007
 * @version 1.0
 *
 * Please create your own implementation.
 * You are allowed to copy all lines you like of this source code, 
 * but you may not modify, compile, or distribute this 'ViewClipDistancer'. 
 * 
 */
public final class ViewClipDistancer extends Behavior {
	
    private View                    sceneView           = 	null;
    private ViewPlatform            viewPlatform        =	null;
    private BranchGroup             sceneBranch         = 	null;
    private TransformGroup          viewPlatformTG      = 	null;

    private Transform3D             vpTransform         = 	new Transform3D();
    private Point3d                 viewPosition        = 	new Point3d();
    private Vector3d                viewTranslation     = 	new Vector3d();

    private WakeupOnTransformChange tgChange            = 	null;

    private Point3d                 sceneCenter         =	new Point3d();
    private double                  distToSceneCenter   = 	0; 
    private double                  sceneRadius         = 	0;
    private double                  radiusFactor        =	1.2; // add 10%-20% 		

    private double                  maxClipRatio        =	1000;	
    private double                  minBackDistance     =	1.0E-5;

    /**
     * Updates a View's back and front clip distances 
     * while the ViewPlatform's distance to the scene is changing.
     * 
     * Requires front and back clip policies as:
     * view.setBackClipPolicy(View.VIRTUAL_EYE);
     * view.setFrontClipPolicy(View.VIRTUAL_EYE);
     * 
     * @param view the target View object
     * @param scene BranchGroup holding the entire scene
     * @param vpTG TransformGroup holding the View's ViewPlatform on which a navigator is operating
     */
    public ViewClipDistancer(View view, BranchGroup scene, TransformGroup vpTG) {

        sceneView = view;
        viewPlatform = view.getViewPlatform();

        sceneBranch = scene;
        viewPlatformTG = vpTG;

        tgChange = new WakeupOnTransformChange(viewPlatformTG);

        // Here optional; don't forget
        this.setSchedulingBounds(new BoundingSphere(new Point3d(), Double.MAX_VALUE));
        sceneBranch.addChild(this);
    }

    @Override
    public void initialize() {
        this.wakeupOn(tgChange);
    }
    /**
     * Called each time when ViewPlatform's transform has changed.
     */
    @Override
    public void processStimulus(java.util.Enumeration criteria) {

        setClipDistances();

        this.wakeupOn(tgChange);
    }

    /**
     * To be called from application as soon as scene is complete and live
     * and each time when scene's extent has changed.
     * Basic parameters for clipping distances calculation will be updated.
     */
    public void updateSceneBounds() {
        if (sceneBranch == null)
            return;
        updateSceneBounds(sceneBranch.getBounds());
    }
    /**
     * 
     * @param sceneBounds arbitrary bounds
     */
    public void updateSceneBounds(Bounds sceneBounds) {		
        if (sceneBounds == null)
            return;

        BoundingSphere sceneSphere = null;
        if (sceneBounds instanceof BoundingSphere) 
            sceneSphere = (BoundingSphere)sceneBounds;
        else
            sceneSphere = new BoundingSphere(sceneBounds);

        sceneSphere.getCenter(sceneCenter);
        sceneRadius = sceneSphere.getRadius() * radiusFactor;

        setClipDistances();
    }

    /**
     * Calculation and update of clipping distances 
     * according to the current bounds.
     */
    private void setClipDistances() {

        if (sceneRadius <= 0)
            return;

        viewPlatform.getLocalToVworld(vpTransform);			
        vpTransform.get(viewTranslation);
        viewPosition.set(viewTranslation);

        distToSceneCenter = sceneCenter.distance(viewPosition);

        double backClipDistance = Math.max(minBackDistance, distToSceneCenter + sceneRadius);
        double frontClipDistance = Math.max(backClipDistance - 2*sceneRadius, 
                                            backClipDistance/maxClipRatio);

        sceneView.setBackClipDistance(backClipDistance);
        sceneView.setFrontClipDistance(frontClipDistance);
    }
}
