/*
 * LICENSE
 *
 * PropellerPuzzleMain.fx is a derived work of JigsawMain.fx
 *
 * Redistribution and use are permitted according to the following license notice.
 *
 * Version: 5.0
 * Date: 2010/09/19
 *
 * Author:
 * August Lammersdorf, InteractiveMesh e.K.
 * Kolomanstrasse 2a, 85737 Ismaning
 * Germany / Munich Area
 * www.InteractiveMesh.com/org
 *
*/

/*
 * JigsawMain.fx
 *
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER
 * Copyright 2009 Sun Microsystems, Inc. All rights reserved. Use is subject to license terms.
 *
 * This file is available and licensed under the following license:
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 *   * Redistributions of source code must retain the above copyright notice,
 *     this list of conditions and the following disclaimer.
 *
 *   * Redistributions in binary form must reproduce the above copyright notice,
 *     this list of conditions and the following disclaimer in the documentation
 *     and/or other materials provided with the distribution.
 *
 *   * Neither the name of Sun Microsystems nor the names of its contributors
 *     may be used to endorse or promote products derived from this software
 *     without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR
 * ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 * ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER
 * Copyright 2008 Sun Microsystems, Inc. All rights reserved. Use is subject to license terms.
 *
 * This file is available and licensed under the following license:
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 *   * Redistributions of source code must retain the above copyright notice,
 *     this list of conditions and the following disclaimer.
 *
 *   * Redistributions in binary form must reproduce the above copyright notice,
 *     this list of conditions and the following disclaimer in the documentation
 *     and/or other materials provided with the distribution.
 *
 *   * Neither the name of Sun Microsystems nor the names of its contributors
 *     may be used to endorse or promote products derived from this software
 *     without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR
 * ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 * ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package com.interactivemesh.j3d.testspace.jfx.propellerpuzzle;

import com.javafx.preview.control.Menu;
import com.javafx.preview.control.MenuItem;
import com.javafx.preview.control.PopupMenu;

import javafx.animation.Interpolator;
import javafx.animation.KeyFrame;
import javafx.animation.Timeline;

import javafx.geometry.HPos;
import javafx.geometry.VPos;

import javafx.scene.Group;
import javafx.scene.Scene;

import javafx.scene.control.Label;
import javafx.scene.control.Separator;
import javafx.scene.control.Slider;
import javafx.scene.control.Tooltip;

import javafx.scene.input.MouseButton;
import javafx.scene.input.MouseEvent;

import javafx.scene.layout.HBox;
import javafx.scene.layout.LayoutInfo;
import javafx.scene.layout.Tile;

import javafx.scene.paint.Color;
import javafx.scene.paint.LinearGradient;
import javafx.scene.paint.Stop;

import javafx.scene.shape.Rectangle;
import javafx.scene.shape.Line;

import javafx.scene.text.Font;
import javafx.scene.text.FontWeight;

import javafx.stage.Screen;
import javafx.stage.Stage;

import javafx.util.Math;

// TODO : How to check for application versus applet ??
// API : getArgument("javafx.applet") will return null if not running as an applet
// API : This is an experimental facility, that may be changed in future versions.
var isApplication: Boolean = (FX.getArgument("javafx.applet") == null);

// Design

def backgroundColor = Color.color(0.0, 0.4, 0.8);

def  labelColor = Color.color(0.8, 0.8, 0.8);
def  hoverColor = Color.color(1.0, 0.7, 0.0);   // orange;
def  pressColor = Color.color(1.0, 0.2, 0.0);   // dark orange;

var textFont: Font;
var menuFont: Font;
var tipFont: Font;

// Puzzle

var fxCanvas3DComp: FXCanvas3DComp;

package var puzzlePieces: PuzzlePieceComponent[];
var puzzleGroup: Group;

var currMaxBoardLength: Integer = 0;
var currBoardLength: Integer = 0;
var pieceLength: Integer = 150;
var puzzleRows: Integer = 0;

var puzzlePieceCount: Integer = 0;
var piecesInPlaceCount: Integer = 0;

var shuffleOrSolveTimeline: Timeline;

// Puzzle board lines
var puzzleLines: Line[];

// Count solved pieces
package function onPieceInPlace(): Void {
    piecesInPlaceCount++;
    if (piecesInPlaceCount == puzzlePieceCount) {
        fxCanvas3DComp.toFront(); // enables 3D mouse navigation on all pieces
        piecesInPlaceCount = 0
    }
}

function resizePuzzleBoard(maxBoardWidth: Integer, maxBoardHeight: Integer): Void {

    if (shuffleOrSolveTimeline != null and shuffleOrSolveTimeline.running)
        shuffleOrSolveTimeline.stop();

    // Update puzzle size
    var maxBoardLength: Number = Math.min(maxBoardWidth, maxBoardHeight);
    if (maxBoardLength < pieceLength) {
        maxBoardLength = pieceLength;
    }

    def newRows: Integer = (maxBoardLength / pieceLength) as Integer;

    puzzleRows = newRows;
    puzzlePieceCount = puzzleRows * puzzleRows;

    currMaxBoardLength = maxBoardLength as Integer;

    def boardLength = puzzleRows * pieceLength;

    if (fxCanvas3DComp.width != boardLength) {
        currBoardLength = boardLength; // 'updatePuzzleBoard' will be called via callback 'fxCanvas3DImageResized'
    }
    else {
        updatePuzzleBoard();
    }
}

function updatePuzzleBoard() {
    // Exchange puzzle pieces
    puzzlePieces = fxCanvas3DComp.createPuzzlePieces(puzzleRows, pieceLength);
    puzzleGroup = Group {
        content: puzzlePieces
    }

    // Create new board lines
    puzzleLines = createPuzzleLines(puzzleRows, puzzleRows);
    // Navigation panel to front
    fxCanvas3DComp.toFront();
}

function createPuzzleLines(rowCount: Integer, colCount: Integer): Line[] {
    
    def rowCtMinus1 = rowCount - 1;
    def colCtMinus1 = colCount - 1;

    var lines: Line[];
    
    insert for (row in [1..rowCtMinus1]) {
        Line {
            stroke: Color.RED
            startX: 5
            startY: (pieceLength * row)
            endX: (pieceLength * colCount) - 5
            endY: (pieceLength * row)
        }
    }
    into lines;

    insert for (col in [1..colCtMinus1]) {
        Line {
            stroke: Color.RED
            startX: (pieceLength * col)
            startY: 5
            endX: (pieceLength * col)
            endY: (pieceLength * rowCount) - 5
        }
    }
    into lines;

    return lines;
}

// Interpolators

var sceneRectWidthInterpolator: Number;

class ShuffleXInterpolator extends Interpolator {
    override public function interpolate(startValue: java.lang.Object, endValue: java.lang.Object, fraction: Number): java.lang.Object {
        // [-0.5, +0.5]
        def random: Number = Math.random() - 0.5;
        def eValue: Number = random * sceneRectWidthInterpolator + (currBoardLength - pieceLength)/2 - Math.signum(random)*(pieceLength/2+5);

        (1-fraction)*(startValue as Number) + fraction * eValue;
    }
}
class ShuffleYInterpolator extends Interpolator {
    override public function interpolate(startValue: java.lang.Object, endValue: java.lang.Object, fraction: Number): java.lang.Object {

        def eValue: Number = Math.random() * currBoardLength -10;

        (1-fraction)*(startValue as Number) + fraction * eValue;
    }
}

//
// Run function
//
function run(args: String[]): Void {

    def screenHeight: Number = Screen.primary.bounds.height;
    def screenDPI: Number = Screen.primary.dpi;

    // screenHeight >= 1200
    var textFontSize = 19;
    var titleFontSize = 34;
    var sceneHeight = 1050;
    var border = 50;            

    // screenHeight  < 1024
    if (screenHeight < 1024) {
        textFontSize = 15;
        titleFontSize = 26;
        border = 30;
        sceneHeight = 700;
        pieceLength = 100;
    }
    // 1024 <= screenHeight < 1200
    else if (screenHeight < 1200) {
        textFontSize = 17;
        titleFontSize = 30;
        border = 40;
        sceneHeight = 1000;
        pieceLength = 120;
    }

    // Bug RT-9312, since 1.3.1
    // Logical fonts (Dialog, Serif, etc.) overruns, although text doesn't exceed the available space.

    def titleFont = Font.font("Amble Cn", FontWeight.REGULAR, titleFontSize);
    textFont = Font.font("Amble Cn", FontWeight.BOLD, textFontSize);
    menuFont = Font.font("Amble Cn", FontWeight.BOLD, textFontSize);
    tipFont = Font.font("Amble Cn", FontWeight.BOLD, textFontSize-2);

    def controlSpace = 10;
    def vertSpace = 3;


    // Stage / Frame
    var stage:Stage;
    var controlBox: HBox;
    
    // Size of scene rectangle -> resize puzzel
    def sceneRectWidth = bind stage.scene.width - 2*border on replace {
        sceneRectWidthInterpolator = sceneRectWidth;
        resizePuzzleBoard(
            sceneRectWidth - 2*border,
            stage.scene.height - (3+2)*border - controlBox.layoutBounds.height - 10,
        );
    };

    var sceneRectHeight = bind stage.scene.height - 3*border on replace {
        resizePuzzleBoard(
            stage.scene.width - (2+2)*border,
            sceneRectHeight - 2*border - controlBox.layoutBounds.height - 10,
        );
    };

    // Initialize puzzle
    currMaxBoardLength = (sceneHeight - 2*border  - 2.5*border) as Integer;
    puzzleRows = (currMaxBoardLength / pieceLength) as Integer;
    puzzlePieceCount = puzzleRows * puzzleRows;
    currBoardLength = puzzleRows * pieceLength;

    // FXCanvas3DImage
    fxCanvas3DComp = FXCanvas3DComp {
        // This panel is used only for receiving mouse events
        opacity: 0.0

        // Set size according to current puzzle board width/height
        layoutInfo: LayoutInfo {width: bind currBoardLength height: bind currBoardLength};

        // Callback: Stage resized => Canvas3D size changed; we are on JavaFX-EDT
        fxCanvas3DImageResized: function(): Void {
            // update puzzle board
            // Wait for Stage resizing is finished
            FX.deferAction(
                function(): Void { updatePuzzleBoard() }
            );

            /* Alternative
            def rTimeline = Timeline {
                repeatCount: 1
                keyFrames: [
                    KeyFrame {
                        time: 100ms;
                        action: function() { updatePuzzleBoard() }
                    }
                ]
            }
            rTimeline.playFromStart();*/
        }

        // Context menu

        // positioning not correctly on this node, doesn't help: 'pickOnBounds: true'
        pickOnBounds: true;
        onMousePressed: function (event) {
            if (popupMenu.showing)
                popupMenu.hide();
        }
        onMouseClicked: function(event) {
            if (event.button == MouseButton.SECONDARY) {
                if (isApplication) {
                    popupMenu.show(fxCanvas3DComp, event.screenX+5, event.screenY);
                }
                else { // Applet workaround
                    popupMenu.show(fxCanvas3DComp, HPos.LEFT, VPos.TOP, event.x, event.y);
                }
            }
        }

    };

    // UniverseFX
    def universeFX = PropellerUniverseFX {
        // Callback of AsyncOperation
        initUniverse: function(universe: PropellerUniverse): Void {

            // Finish FXCanvas3DComp
            fxCanvas3DComp.initFXCanvas3D(universe);

            // Init puzzle board
            resizePuzzleBoard(currMaxBoardLength, currMaxBoardLength);

            // Show frame
            stage.visible = true;
        }

        // Change rotation speed/direction
        rotationValue: bind rotationSlider.value as Integer

    };

    // Shuffle / Solve animation
    def shuffleXInterpolator = ShuffleXInterpolator {};
    def shuffleYInterpolator = ShuffleYInterpolator {};

    //
    // Controls
    //

    def shuffleLabel = ButtonLabel {
        text: "Shuffle"
        tooltip: PuzzleTooltip { text: "Shuffles the puzzle pieces" }
        action: function(): Void {
            if (shuffleOrSolveTimeline != null and shuffleOrSolveTimeline.running)
                shuffleOrSolveTimeline.stop();

            // Shuffle animation
            def shuffleTimeline = Timeline {
                repeatCount: 1
                keyFrames: [
                    KeyFrame {
                        time: 0s;
                        values:
                        for (piece in puzzlePieces) {
                            def temp = piece as PuzzlePieceComponent;
                            [temp.inCorrectPlace => false,
                             temp.translateX => temp.translateX,
                             temp.translateY => temp.translateY]
                        }
                    },
                    KeyFrame {
                        time: 2s;
                        values:
                        for (piece in puzzlePieces) {
                            def temp = piece;
                            def random = Math.random() - 0.5;
                            [temp.translateX => (
                                random * sceneRectWidth + (currBoardLength - pieceLength)/2 - Math.signum(random)*(pieceLength/2+5)
                                ) tween shuffleXInterpolator,
                             temp.translateY => (
                                Math.random() * (currBoardLength) -10) tween shuffleYInterpolator]
                        }
                    }
                ]
            };

            piecesInPlaceCount = 0;
            fxCanvas3DComp.toBack();
  
            shuffleOrSolveTimeline = shuffleTimeline;
            shuffleOrSolveTimeline.playFromStart();
        }
    }
    def solveLabel = ButtonLabel {
        text: "Solve"
        tooltip: PuzzleTooltip { text: "Solves the puzzle" }
        action: function(): Void {
            if (shuffleOrSolveTimeline != null and shuffleOrSolveTimeline.running)
                shuffleOrSolveTimeline.stop();

            // Solve animation
            def solveTimeline = Timeline {
                repeatCount: 1
                keyFrames: [
                    KeyFrame {
                        time: 0s;
                        values:
                        for (piece in puzzlePieces){
                            def temp = piece;
                            [temp.translateX => temp.translateX,
                             temp.translateY => temp.translateY]
                        }
                    },
                    KeyFrame {
                        time: 2s;
                        values:
                        for (piece in puzzlePieces){
                            def temp = piece as PuzzlePieceComponent;
                            [temp.translateX => temp.locationX,
                             temp.translateY => temp.locationY,
                             temp.inCorrectPlace => true]
                        }
                    }
                ]
            };

            piecesInPlaceCount = 0;
            fxCanvas3DComp.toFront();

            shuffleOrSolveTimeline = solveTimeline;
            shuffleOrSolveTimeline.playFromStart();
        }
    }

    // Rotation

    def actionStopRotation = function(): Void { 
        rotationSlider.value = 0;
        fxCanvas3DComp.fpsPaint = 0;
    };
    def rotationLabel = BaseLabel { // 3 = 60000ms/( 200ms min duration * 100 max value )
        text: bind "RPM : {(rotationSlider.value * 3) as Integer}"
        tooltip: PuzzleTooltip { text: "Revolution per minute" }
    };
    def rotationSlider = Slider {
        blockIncrement: 1
        min: 0
        max: 100
        value: 0

        tooltip: PuzzleTooltip { text: bind "Revolution per minute" }
    }

    // VantagePoints
    
    def actionVpFront = function(): Void { universeFX.setVantagePoint("Front") };
    def actionVpTop = function(): Void { universeFX.setVantagePoint("Top") };
    def actionVpBack = function(): Void { universeFX.setVantagePoint("Back") };
    def actionVpAxis = function(): Void { universeFX.setVantagePoint("Axis") };
    def actionVpPiston = function(): Void { universeFX.setVantagePoint("Piston") };
    def actionVpPistonrod = function(): Void { universeFX.setVantagePoint("Pistonrod") };

    def frontVpButton = ButtonLabel {
        text: "Front"
        tooltip: PuzzleTooltip { text: "Front viewpoint" }
        action: actionVpFront
    };
    def backVpButton = ButtonLabel {
        text: "Back"
        tooltip: PuzzleTooltip { text: "Back viewpoint" }
        action: actionVpBack
    };
    def axisVpButton = ButtonLabel {
        text: "Prop. Shaft"
        tooltip: PuzzleTooltip { text: "Propeller shaft viewpoint" }
        action: actionVpAxis
    };
    def pistonVpButton = ButtonLabel {
        text: "  Piston  "
        tooltip: PuzzleTooltip { text: "Inside piston viewpoint" }
        action: actionVpPiston
    };
    def pistonrodVpButton = ButtonLabel {
        text: "Piston Rod"
        tooltip: PuzzleTooltip { text: "Piston rod viewpoint" }
        action: actionVpPistonrod
    };
    def topVpButton = ButtonLabel {
        text: "Top"
        tooltip: PuzzleTooltip { text: "Top viewpoint" }
        action: actionVpTop
    };

    // Puzzle piece size

    def pieceSizeLabel = BaseLabel {
        text: bind "Piece Size : {pieceSizeSlider.value as Integer}"
        tooltip: PuzzleTooltip { text: "Puzzle piece size : [75, 200] pixels" }
    }
    def pieceSizeSlider: Slider = Slider {
        blockIncrement: 1
        min: 75
        max: 200
        value: 150
        tooltip: PuzzleTooltip { text: "Puzzle piece size : [75, 200] pixels" }

        var pressedValue: Integer

        onMousePressed: function(event: MouseEvent) {
            pressedValue = pieceSizeSlider.value as Integer;
        }
        onMouseReleased: function(event: MouseEvent) {
            if (pressedValue != (pieceSizeSlider.value as Integer)) {
                pieceLength = pieceSizeSlider.value as Integer;
                if (currMaxBoardLength > 0 and pieceLength > 0)
                    resizePuzzleBoard(currMaxBoardLength, currMaxBoardLength);
            }
        }
    }

    // Frames per second

    def fpsLabel = BaseLabel {
        text: "F P S"
        tooltip: PuzzleTooltip { text: "Frame per second ( <= 33 )" }
    }
    def fpsValueLabel = BaseLabel {
        text: bind String.valueOf(fxCanvas3DComp.fpsPaint)
        tooltip: PuzzleTooltip { text: "Frame per second ( <= 33 )" }
   }

    //
    // Control Box
    //
    controlBox = HBox {
        var tile: Tile;
        var tile1: Tile;
        var lInfo = LayoutInfo {width: controlSpace*3 height: bind tile.height};
        var tile0: Tile;

        content: [
            // VantagePoints
            Tile {
                hgap: vertSpace vgap: vertSpace columns: 3
                content: [
                    frontVpButton, topVpButton, backVpButton,
                    axisVpButton, pistonVpButton, pistonrodVpButton
                ]
            },
            Separator {vertical: true layoutInfo: lInfo},
            // Rotation
            tile = Tile {
                hgap: 0 vgap: vertSpace columns: 1
                content: [rotationLabel, rotationSlider]
            },
            Separator {vertical: true layoutInfo: lInfo},
            // Shuffle, solve
            tile1 = Tile {
                hgap: 0 vgap: vertSpace columns: 1
                content: [shuffleLabel, solveLabel]
            },
            Separator {vertical: true layoutInfo: lInfo},
            Tile {
                hgap: 0 vgap: vertSpace columns: 1
                content: [pieceSizeLabel, pieceSizeSlider]
            },
            Separator {vertical: true layoutInfo: lInfo},
            Tile {
                autoSizeTiles: false
                tileWidth: bind tile1.tileWidth*0.8
                hgap: 0 vgap: vertSpace columns: 1
                content: [fpsLabel, fpsValueLabel]
            }
        ]
    }

    //
    // PopupMenu
    //
    def popupMenu = PopupMenu {

        var menuItem: MenuItem
        layoutInfo: LayoutInfo {height: bind menuItem.layoutBounds.height * (if (isApplication) 4 else 3) }

        items: [
            PopupSubMenu {
                text: "Select viewpoint"
                items: [
                    //PopupMenuItem { text: "Front" action: actionVpFront } TODO ??!!
                    PopupMenuItem { text: "Front" action: function() { actionVpFront.invoke() } },
                    PopupMenuItem { text: "Top" action: function() { actionVpTop.invoke() } },
                    PopupMenuItem { text: "Back" action: function() { actionVpBack.invoke() } },
                    PopupMenuItem { text: "Prop. Shaft" action: function() { actionVpAxis.invoke() } }
                    PopupMenuItem { text: "Piston" action: function() { actionVpPiston.invoke() } }
                    PopupMenuItem { text: "Piston Rod" action: function() { actionVpPistonrod.invoke() } }
                ]
            },
            Separator {},
            menuItem = PopupMenuItem {
                text: "Stop rotation"
                //action: actionStopRotation TODO ??!!
                action: function() { actionStopRotation.invoke() }
            }
        ]

        override var onAction = function(item :MenuItem):Void { hide() }

        visible: false
    }
    // Complete popupMenu
    if (isApplication) {
        insert Separator{} into popupMenu.items;
        insert PopupMenuItem {
            text: "Exit application"
            action: function() { stage.close() } 
        } into popupMenu.items;
    }

    // Headline
    def headLine = BaseLabel {
        text: "Java 3D meets JavaFX"
        font: titleFont
    };

    //
    // Stage / scene
    //
    stage = Stage {
        title: "InteractiveMesh : FXPropellerPuzzle"
        visible: false
        resizable: true
        onClose: function() {
            universeFX.closeUniverse();
        }
        scene: Scene {
            fill: backgroundColor
            // Start size
            width: sceneHeight * 1.25
            height: sceneHeight + border

            var popupPanel: Rectangle;

            content: [
                // Context menu
                popupMenu,

                // Background node for popup menu
                popupPanel = Rectangle {
                    width: bind stage.scene.width
                    height: bind stage.scene.height
                    fill: backgroundColor
                    onMousePressed: function (event) {
                        if (popupMenu.showing)
                            popupMenu.hide();
                    }
                    onMouseClicked: function(event) {
                        if (event.button == MouseButton.SECONDARY) {
                            if (isApplication) {
                                popupMenu.show(popupPanel, event.screenX+5, event.screenY);
                            }
                            else { // Applet workaround
                                popupMenu.show(popupPanel, HPos.LEFT, VPos.TOP, event.x, event.y);
                            }
                        }
                    }
                },

                // Headline
                Group {
                    layoutX: border + controlSpace
                    layoutY: bind border - headLine.layoutBounds.height * 0.5
                    content: headLine
                },

                // LinearGradient background
                Rectangle {
                    layoutX: border
                    layoutY: 2 * border
                    width: bind sceneRectWidth
                    height: bind sceneRectHeight 
                    fill: LinearGradient {
                        startX: 0.0, startY: 0.0, endX: 0.0, endY: 1.0
                        proportional: true
                        stops: [ Stop { offset: 0.0 color: backgroundColor },
                                 Stop { offset: 0.75 color: Color.rgb(0, 153, 255) },
                                 Stop { offset: 1.0 color: backgroundColor } ]
                    }
                },

                // Puzzle board
                Group {
                    layoutX: bind border + (sceneRectWidth - currBoardLength)/2
                    layoutY: 3*border + 3
                    content: [
                        // Rectangle to place pieces in
                        Rectangle {
                            layoutX: -3
                            layoutY: -3
                            width: bind currBoardLength + 6
                            height: bind currBoardLength + 6
                            opacity: 0.15
                            fill: Color.rgb(80, 80, 80);
                            stroke: Color.rgb(0, 0, 205);
                            strokeWidth: 3
                        },
                        // Lines
                        Group {
                            content: bind puzzleLines
                        },

                        // Puzzle pieces
                        Group {
                            // fxCanvas3DComp.toFront()/toBack() (for 3D mouse navigation)
                            content: [
                                // transparent
                                fxCanvas3DComp,
                                Group {
                                    content: bind puzzleGroup
                                }
                            ]
                        }
                    ]
                },
                
                // Puzzle controls
                Group {
                    layoutX: bind border + (sceneRectWidth - controlBox.layoutBounds.width) * 0.5
                    layoutY: bind 2 * border + sceneRectHeight - controlBox.layoutBounds.height - border*0.5
                    content: [
                        // Control background
                        Rectangle {
                            layoutX: -10
                            layoutY: -10
                            width: bind controlBox.layoutBounds.width + 20
                            height: bind controlBox.layoutBounds.height + 20
                            //opacity: 0.15
                            fill: LinearGradient {
                                startX: 0.0, startY: 0.0, endX: 0.0, endY: 1.0
                                proportional: true
                                stops: [ Stop { offset: 0.0 color: Color.rgb(0, 153, 255) },
                                         Stop { offset: 0.4 color: backgroundColor },
                                         Stop { offset: 0.8 color: backgroundColor },
                                         Stop { offset: 1.0 color: Color.rgb(0, 128, 228) } ]
                            }
                        },

                        controlBox
                    ]
                }                
            ]
        }
    }
    
    //
    // Start
    //
    // JavaTaskBase
    universeFX.start();

} // End of run function

//
// Control Classes
//

class PopupSubMenu extends Menu {
    override var font = menuFont
}
class PopupMenuItem extends MenuItem {
    override var font = menuFont
}

class PuzzleTooltip extends Tooltip {
    override var font = tipFont;
}

def fillLayoutInfo =  LayoutInfo{hfill: true vfill: true};

class BaseLabel extends Label {
    override var font = textFont;
    override var textFill = labelColor;
    override var hpos = HPos.CENTER;
    override var layoutInfo = fillLayoutInfo;

    override var onMouseEntered = function(event: MouseEvent) {
        if (tooltip != null) tooltip.activate();
    }
    override var onMouseExited = function(event: MouseEvent) {
        if (tooltip != null) tooltip.deactivate();
    }
    override var onMousePressed = function(event: MouseEvent) {
        if (tooltip != null) tooltip.deactivate();
    }
}

class ButtonLabel extends BaseLabel {

    var action: function(): Void;

    override var onMouseEntered = function(event: MouseEvent) {
        if (not pressed) {
            textFill = hoverColor;
        }
        if (tooltip != null) tooltip.activate();
    };
    override var onMouseExited = function(event: MouseEvent) {
        if (not pressed) {
            textFill = labelColor;
        }
        if (tooltip != null) tooltip.deactivate();
    };
    override var onMousePressed = function(event: MouseEvent) {
        textFill = pressColor;
        if (tooltip != null) tooltip.deactivate();
    };
    override var onMouseReleased = function(event: MouseEvent) {
        if (disabled)
            return;

        if (hover) {
            action();
            textFill = hoverColor;
        }
        else {
            textFill = labelColor;
        }
    };
}
