
/*
 * j3d-examples: src.resources.images.bg.jpg
 *
 * Copyright (c) 2007 Sun Microsystems, Inc. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * - Redistribution of source code must retain the above copyright
 *   notice, this list of conditions and the following disclaimer.
 *
 * - Redistribution in binary form must reproduce the above copyright
 *   notice, this list of conditions and the following disclaimer in
 *   the documentation and/or other materials provided with the
 *   distribution.
 *
 * Neither the name of Sun Microsystems, Inc. or the names of
 * contributors may be used to endorse or promote products derived
 * from this software without specific prior written permission.
 *
 * This software is provided "AS IS," without a warranty of any
 * kind. ALL EXPRESS OR IMPLIED CONDITIONS, REPRESENTATIONS AND
 * WARRANTIES, INCLUDING ANY IMPLIED WARRANTY OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE OR NON-INFRINGEMENT, ARE HEREBY
 * EXCLUDED. SUN MICROSYSTEMS, INC. ("SUN") AND ITS LICENSORS SHALL
 * NOT BE LIABLE FOR ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF
 * USING, MODIFYING OR DISTRIBUTING THIS SOFTWARE OR ITS
 * DERIVATIVES. IN NO EVENT WILL SUN OR ITS LICENSORS BE LIABLE FOR
 * ANY LOST REVENUE, PROFIT OR DATA, OR FOR DIRECT, INDIRECT, SPECIAL,
 * CONSEQUENTIAL, INCIDENTAL OR PUNITIVE DAMAGES, HOWEVER CAUSED AND
 * REGARDLESS OF THE THEORY OF LIABILITY, ARISING OUT OF THE USE OF OR
 * INABILITY TO USE THIS SOFTWARE, EVEN IF SUN HAS BEEN ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGES.
 *
 * You acknowledge that this software is not designed, licensed or
 * intended for use in the design, construction, operation or
 * maintenance of any nuclear facility.
 *
 */

package com.interactivemesh.j3d.testspace.jfx.propellerpuzzle;

import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.Dimension;

import java.awt.image.BufferedImage;

import java.io.IOException;

import java.net.URL;

import java.util.Enumeration;
import java.util.HashMap;
import java.util.LinkedHashMap;

import javax.imageio.ImageIO;

import javax.media.j3d.Alpha;
import javax.media.j3d.Background;
import javax.media.j3d.Behavior;
import javax.media.j3d.BoundingSphere;
import javax.media.j3d.Bounds;
import javax.media.j3d.BranchGroup;
import javax.media.j3d.Canvas3D;
import javax.media.j3d.DirectionalLight;
import javax.media.j3d.GraphicsConfigTemplate3D;
import javax.media.j3d.Group;
import javax.media.j3d.ImageComponent2D;
import javax.media.j3d.Locale;
import javax.media.j3d.PhysicalBody;
import javax.media.j3d.PhysicalEnvironment;
import javax.media.j3d.SceneGraphObject;
import javax.media.j3d.Shape3D;
import javax.media.j3d.Transform3D;
import javax.media.j3d.TransformGroup;
import javax.media.j3d.View;
import javax.media.j3d.ViewPlatform;
import javax.media.j3d.VirtualUniverse;
import javax.media.j3d.WakeupCriterion;
import javax.media.j3d.WakeupOnBehaviorPost;
import javax.media.j3d.WakeupOnElapsedFrames;
import javax.media.j3d.WakeupOr;

import javax.swing.JPanel;

import javax.vecmath.Color3f;
import javax.vecmath.Point3d;
import javax.vecmath.Vector3d;

// X3DModelImporter
import com.interactivemesh.j3d.interchange.ext3d.XImportException;
import com.interactivemesh.j3d.interchange.ext3d.XShapeReader;

// FXCanvas3DImage 1.0, see http://www.interactivemesh.org/testspace/j3dmeetsjfx.html
import com.interactivemesh.j3d.community.gui.FXCanvas3DImage;
import com.interactivemesh.j3d.community.gui.FXCanvas3DImageRepainter;

// OrbitBehaviorInterim 2.1, see http://www.interactivemesh.org/testspace/orbitbehavior.html
import com.interactivemesh.j3d.community.utils.navigation.orbit.OrbitBehaviorInterim;

// JavaFX async operations
import javafx.async.RunnableFuture;

/**
 * PropellerUniverse.java
 *
 * Version: 5.0
 * Date: 2010/09/19
 *
 * Copyright (c) 2009-2010
 * August Lammersdorf, InteractiveMesh e.K.
 * Kolomanstrasse 2a, 85737 Ismaning
 * Germany / Munich Area
 * www.InteractiveMesh.com/org
 *
 * Please create your own implementation.
 * This source code is provided "AS IS", without warranty of any kind.
 * You are allowed to copy and use all lines you like of this source code
 * without any copyright notice,
 * but you may not modify, compile, or distribute this 'PropellerUniverse.java'.
 *
 * See also license notice above: j3d-examples: src.resources.images.bg.jpg
 */
final class PropellerUniverse implements RunnableFuture {

    static {
        System.out.println("FXPropellerPuzzle : Copyright (c) 2009-2010 August Lammersdorf, www.InteractiveMesh.com.");
    }

    private BoundingSphere      globalBounds        = 	null;

    private View                view                = 	null;
    private Locale              locale              = 	null;

    private OrbitBehaviorInterim orbitBehInterim    = 	null;

    private BranchGroup         sceneBranch         = 	null;
    private BranchGroup         viewBranch          = 	null;
    private BranchGroup         enviBranch          = 	null;

    private TransformGroup      crankarmGroup       =   null;
    private TransformGroup      cylinderGroup       =   null;
    private TransformGroup      propellerGroup      =   null;

    private PropellerBehavior2  propellerBehavior2  =   null;
    private Alpha               propellerAlpha      =   null;
    private long                propellerStartTime  =   0;

    private Color               bgColor0            =   new Color(0, 102, 204); // RGB 0.0f, 0.4f, 0.8f, HSB 209, 100,  80

    private VantagePointBehavior vpExecutor         =	null;
    private LinkedHashMap<String, VantagePoint> vantagepointHM = new LinkedHashMap<String, VantagePoint>();


    PropellerUniverse() {
    }
    
    // JavaFX Interface RunnableFuture
    @Override
    public void run() {
        initUniverse();
    }

    // 
    FXCanvas3DImage createFXCanvas3D(FXCanvas3DImageRepainter repainter, JPanel parent) {

        FXCanvas3DImage fxCanvas3D = null;

        try {
            GraphicsConfigTemplate3D gCT = new GraphicsConfigTemplate3D();

            fxCanvas3D = new FXCanvas3DSquareImage(gCT);
            fxCanvas3D.setRepainter(repainter);
        }
        catch (Exception e) {
            System.out.println("PropellerUniverse: FXCanvas3DImage failed !!");
            e.printStackTrace();
            System.exit(0);
        }

        // Optional: less 'flickering' during resizing
        fxCanvas3D.setBackground(bgColor0);

        // Due to Java 3D's implementation of off-screen rendering:
        // 1. Set size
        // 2. Provide a parent
        // 3. Get the heavyweight off-screen Canvas3D and add this to the view object

        // 1. Size
        Dimension dim = parent.getSize();
        if (dim.width < 1 || dim.height < 1) {
            dim.width = 100;
            dim.height = 100;
            parent.setSize(dim);
        }
        parent.setPreferredSize(dim);
        fxCanvas3D.setPreferredSize(dim);
        fxCanvas3D.setSize(dim);

        // 2. Parent
        // BorderLayout
        parent.setLayout(new BorderLayout());
        parent.add(fxCanvas3D, BorderLayout.CENTER);

        // 3. Heavyweight off-screen Canvas3D of the lightweight FXCanvas3D
        Canvas3D offCanvas3D = fxCanvas3D.getOffscreenCanvas3D();
        if (offCanvas3D != null) {
            // View renders into the off-screen Canvas3D
            view.addCanvas3D(offCanvas3D); 
        }
        else {
            System.out.println("PropellerUniverse: Off-screen Canvas3D is null !!");
            System.exit(0);
        }

        setupNavigator(fxCanvas3D);

        return fxCanvas3D;
    }

    //
    // Scene interaction
    //

    void setVantagePoint(String name) {
        VantagePoint vp = vantagepointHM.get(name);
        vpExecutor.setVP(vp);
        vpExecutor.postId(VantagePointBehavior.APPLY_VP);
    }

    void setRotationSpeed(int speed) {
        propellerBehavior2.rotationSpeed(speed);
    }

    void closeUniverse() {
        view.removeAllCanvas3Ds();
        view.attachViewPlatform(null);
        locale.getVirtualUniverse().removeAllLocales();
    }

    //
    // Create universe
    //
    private void initUniverse() {

        createVantagePoints();

        createUniverse();
        createScene();

        setLive();

        setVantagePoint("Back");
    }

    // Setup navigation
    private void setupNavigator(FXCanvas3DImage component) {

        orbitBehInterim.setAWTComponent(component);

        double sceneRadius = 1.0f;

        Bounds bounds = sceneBranch.getBounds();
        BoundingSphere sphereBounds = null;

        if (bounds.isEmpty()) {
            sphereBounds = new BoundingSphere();
        }
        else {
            if (bounds instanceof BoundingSphere)
                sphereBounds = (BoundingSphere)bounds;
            else
                sphereBounds = new BoundingSphere(bounds);
        }

        sceneRadius = sphereBounds.getRadius();

        orbitBehInterim.setTransFactors(sceneRadius/4.0f, sceneRadius/4.0f);
        orbitBehInterim.setZoomFactor(sceneRadius/4.0f);
        orbitBehInterim.setRotFactors(0.4f, 0.4f);

        orbitBehInterim.setClippingBounds(sphereBounds);
    }

    private void createScene() {
        
        // PropellerModelOCC-AllShapes.x3d -> file only includes named Shape nodes
        
        XShapeReader x3dImporter = new XShapeReader();
        try {
            x3dImporter.read(this.getClass().getResource("resources/PropellerModelOCC-AllShapes.x3d"));
        }
        catch (XImportException e) {
            e.printStackTrace();
            return;
        }
        
        // Typesafe retrieving of shapes (no check if null)
        
        Shape3D[] x3dShapes = x3dImporter.getImport();
        
        HashMap<String, Shape3D> namedShapes = new HashMap<String, Shape3D>();
        for (Shape3D shape : x3dShapes) {
            namedShapes.put(shape.getName(), shape);
        }
        
        // Alternative :
        // Map<String, SceneGraphObject> namedSGOs = x3dImporter.getNamedSGOs();   
        // Shape3D crankarm = (Shape3D)namedSGOs.get("CrankArm")        
        // etc.

        propellerGroup = new TransformGroup();
        propellerGroup.setCapability(TransformGroup.ALLOW_TRANSFORM_WRITE);
        TransformGroup propPositionGroup = new TransformGroup();
        Transform3D t3d = new Transform3D();
        t3d.rotZ(Math.toRadians(45));
        propPositionGroup.setTransform(t3d);
        propPositionGroup.addChild(namedShapes.get("Propeller"));
        propellerGroup.addChild(propPositionGroup);
        propellerGroup.addChild(namedShapes.get("PropAxis"));
       
        crankarmGroup = new TransformGroup();
        crankarmGroup.setCapability(TransformGroup.ALLOW_TRANSFORM_WRITE);
        crankarmGroup.addChild(namedShapes.get("Crankarm"));
        
        cylinderGroup = new TransformGroup();
        cylinderGroup.setCapability(TransformGroup.ALLOW_TRANSFORM_WRITE);
        cylinderGroup.addChild(crankarmGroup);
        cylinderGroup.addChild(namedShapes.get("PistonWristPin"));
        cylinderGroup.addChild(namedShapes.get("Piston"));
        
        Group engineblockGroup = new Group();
        engineblockGroup.addChild(namedShapes.get("EngineBlock"));
        engineblockGroup.addChild(namedShapes.get("CylinderHead"));
   
        // Propeller assembly
        Group modelGroup = new Group();
        modelGroup.addChild(propellerGroup);
        modelGroup.addChild(cylinderGroup);
        modelGroup.addChild(engineblockGroup);

        // Rotation : PropellerBehavior
        propellerAlpha = new Alpha();
        propellerAlpha.setLoopCount(-1);
        // Initialize rotation
        propellerAlpha.setIncreasingAlphaDuration(600 * 47);    // start speed = min speed
        propellerStartTime = System.currentTimeMillis();        // start time
        propellerAlpha.setStartTime(propellerStartTime);        // start
        propellerAlpha.pause(propellerStartTime);               // pause at start time

        propellerBehavior2 = new PropellerBehavior2();
        propellerBehavior2.setSchedulingBounds(globalBounds);

        // Add to scene
        sceneBranch.addChild(propellerBehavior2);
        sceneBranch.addChild(modelGroup);
    }
    
    // Set live
    private void setLive() {
        locale.addBranchGraph(sceneBranch);
        locale.addBranchGraph(viewBranch);
        locale.addBranchGraph(enviBranch);
    }

	// Base world
    private void createUniverse() {

        // Bounds
    	globalBounds = new BoundingSphere();
        globalBounds.setRadius(Double.MAX_VALUE);

        //
        // Viewing
        //
        view = new View();
        view.setPhysicalBody(new PhysicalBody());
        view.setPhysicalEnvironment(new PhysicalEnvironment());
        // limit frames per second : so JavaFX loop runs with 'no' interruptions
        view.setMinimumFrameCycleTime(30);

        //
        // SuperStructure
        //
        VirtualUniverse vu = new VirtualUniverse();
        locale = new Locale(vu);

        //
        // BranchGraphs
        //
        sceneBranch = new BranchGroup();
        viewBranch = new BranchGroup();
        enviBranch = new BranchGroup();

        // ViewBranch

        TransformGroup viewTG = new TransformGroup();
        viewTG.setCapability(TransformGroup.ALLOW_TRANSFORM_WRITE);

        ViewPlatform vp = new ViewPlatform();
        view.attachViewPlatform(vp);

        orbitBehInterim = new OrbitBehaviorInterim(OrbitBehaviorInterim.REVERSE_ALL);
        orbitBehInterim.setViewingTransformGroup(viewTG);
        orbitBehInterim.setVpView(view);
        orbitBehInterim.setSchedulingBounds(globalBounds);
        orbitBehInterim.setClippingEnabled(true);

        vpExecutor = new VantagePointBehavior(orbitBehInterim);
        vpExecutor.setSchedulingBounds(globalBounds);

        DirectionalLight headLight = new DirectionalLight();
        headLight.setInfluencingBounds(globalBounds);

        viewTG.addChild(vp);
        viewTG.addChild(orbitBehInterim);
        viewTG.addChild(vpExecutor);
        viewTG.addChild(headLight);

        viewBranch.addChild(viewTG);

        // EnviBranch

        Background bg = new Background();
        bg.setApplicationBounds(globalBounds);
        bg.setColor(new Color3f(bgColor0));

        URL imageUrl = this.getClass().getResource("resources/bg.jpg");
        try {
            BufferedImage bgImage = ImageIO.read(imageUrl);
            if (bgImage != null) {
                bg.setImageScaleMode(Background.SCALE_FIT_ALL);
                bg.setImage(new ImageComponent2D(ImageComponent2D.FORMAT_RGB, bgImage, false, false));
            }
        }
        catch (IOException e) {
        }
        
        enviBranch.addChild(bg);
    }

    private void createVantagePoints() {
        
        Point3d  lookAt    = new Point3d(0, 0, -0.0525);
        Vector3d upVector  = new Vector3d(0.0, 1.0, 0.0);
        Point3d  rotCenter = new Point3d(0, 0, -0.0525);
        //                name   -  VP's position         -                  look at           -               up vector         -          center of rotation
        new VantagePoint("Front",     new Point3d(0, 0, 0.9),              lookAt,                         upVector,                    rotCenter);
        new VantagePoint("Back",      new Point3d(0, 0, -0.88),            lookAt,                         upVector,                    rotCenter);
        new VantagePoint("Top",       new Point3d(0, 0.665, -0.0525),      lookAt,                         new Vector3d(0.0, 0.0, 1.0), new Point3d(0, 0, -0.0525));

        new VantagePoint("Axis",      new Point3d(0, 0, -0.31),            lookAt,                         upVector,                    rotCenter);
        new VantagePoint("Pistonrod", new Point3d(-0.159, 0, -0.268),      new Point3d(0.045, 0, -0.1275), upVector,                    new Point3d(0.045, 0, -0.1275));
        new VantagePoint("Piston",    new Point3d(0.1087, 0.023, -0.1038), new Point3d(0, 0, -0.10378),    upVector,                    new Point3d(0, 0, -0.10378));

    }

    private final class VantagePoint {

        private String      name        =   null;
        private Point3d     position    =   new Point3d();
        private Point3d     viewCenter 	=   new Point3d();
        private Vector3d    up          =   new Vector3d();
        private Point3d     rotCenter 	=   new Point3d();

        private VantagePoint(String name,
                             Point3d position,
                             Point3d viewCenter,
                             Vector3d up,
                             Point3d rotationCenter) {
            this.name = name;

            this.position.set(position);
            this.viewCenter.set(viewCenter);
            this.up.set(up);
            this.rotCenter.set(rotationCenter);

            vantagepointHM.put(name, this);
        }

        @Override
        public String toString() {
            return name;
        }

        private void applyTo(OrbitBehaviorInterim navigator) {            
            navigator.setViewingTransform(position, viewCenter, up, rotCenter);
	}
    }

    // Sets vantage point in behavior scheduler
    private final class VantagePointBehavior extends Behavior {

        private static final int        APPLY_VP = 1;
        private WakeupOnBehaviorPost 	post = new WakeupOnBehaviorPost(this, APPLY_VP);

        private OrbitBehaviorInterim 	orbitBeh = null;
        private VantagePoint            vantagePoint = null;

        private VantagePointBehavior(OrbitBehaviorInterim orbitBeh) {
            this.orbitBeh = orbitBeh;
        }

        private void setVP(VantagePoint vp) {
            vantagePoint = vp;
        }

        @Override
        public void initialize() {
            wakeupOn(post);
        }
        @Override
        public void processStimulus(Enumeration criteria) {
            if (vantagePoint != null) {
                vantagePoint.applyTo(orbitBeh);
            }

            vantagePoint = null;
            wakeupOn(post);
        }
    }
    
    private class PropellerBehavior2 extends Behavior {

        // Model
        private final double onePi      = Math.PI;
        private final double twoPi      = Math.PI * 2;

        private double axisRadians       = 0;
        private double sinAxisRadians    = 0;
        private double crankarmRadians 	 = 0;

        private double radius            = 0.045;
        private double crankarmLength 	 = 0.120;
        private double dist              = 0.165; // 0.045 + 0.120
        private double currDist          = 0.165;
        private double transX            = 0;

        // Transformation
        private Transform3D cylinderTransform   = new Transform3D();
        private Vector3d cylinderTranslation    = new Vector3d();

        private Transform3D propellerRotation   = new Transform3D();

        private Transform3D crankarmTransform   = new Transform3D();
        private Transform3D crankarmRotation    = new Transform3D();
        private Transform3D crankarmCenterTrans = new Transform3D();
    
        private Transform3D crankarmCenterTransInv  =   new Transform3D();
        private Transform3D crankarmTemp            =   new Transform3D();

        // "Alpha"
        private long NsecPerMsec        = 1000000; // nano sec per micro sec
        private long MinDuration        = 200L;    // 0.2 sec per rotation
        private long startTime          = 0L;
        private double duration         = 0;
        private double currAlphaValue   = 0;
        private double loops            = 0;

        // WakeupCriterions
        private WakeupOnBehaviorPost speedPost = new WakeupOnBehaviorPost(this, 0);
        private WakeupOr activeFramesOrPost = null;

        private PropellerBehavior2() {
            crankarmCenterTrans.setTranslation(new Vector3d(-0.165, 0, 0));
            crankarmCenterTransInv.setTranslation(new Vector3d(0.165, 0, 0));
        
            WakeupCriterion[] cons = {new WakeupOnBehaviorPost(this, 0), new WakeupOnElapsedFrames(0)};
            activeFramesOrPost = new WakeupOr(cons);
        }

        // Speed, s = [0, 100]
        private void rotationSpeed(int s) {
            postId(s);
        }

        //
        // Behavior
        //

        @Override
        public void initialize() {
            wakeupOn(speedPost);
        }

        @Override
        public void processStimulus(Enumeration criteria) {
            while (criteria.hasMoreElements()) {
                WakeupCriterion criterion = (WakeupCriterion)criteria.nextElement();
                if (criterion instanceof WakeupOnBehaviorPost) {
                    int speed = ((WakeupOnBehaviorPost)criterion).getTriggeringPostId();
                    // Stop rotation
                    if (speed == 0) {
                        wakeupOn(speedPost);
                        return;
                    }
                    // Set speed: slider [1, 100] -> [10, 0.1] sec per rotation
                    else {

                        // duration -> speed (long)
                        duration = (NsecPerMsec * MinDuration * 100f/speed);
                        // startTime -> sync  / -20,000,000 (-20 millisec) : jump to next alphaVlaue, results in smoother acceleration
                        startTime = (long)(System.nanoTime() - (currAlphaValue * duration) -20000000);

                        wakeupOn(activeFramesOrPost);
                    }
                }
                else if (criterion instanceof WakeupOnElapsedFrames) {
                    // Rotation

                    // [0, 1] : fraction of current loop
//                  currAlphaValue = ((System.nanoTime() - startTime) % duration) / duration;
                    loops = (System.nanoTime() - startTime);
                    loops /= duration;
                    currAlphaValue = loops - (int)loops;

                    // [0, twoPi]
                    axisRadians = twoPi * currAlphaValue;

                    sinAxisRadians = Math.sin(axisRadians);

                    // Law of sines

                    crankarmRadians = Math.asin(radius * sinAxisRadians / crankarmLength);

                    // If currRadians != onePi | twoPi | 0
                    if (sinAxisRadians != 0.0) {
                        currDist = (crankarmLength / sinAxisRadians * Math.sin(onePi-axisRadians-crankarmRadians));
                    }
                    else {
                        if (axisRadians == onePi)
                            currDist = crankarmLength - radius;
                        else
                            currDist = crankarmLength + radius;
                    }

                    transX = dist - currDist;

                    cylinderTranslation.setX(-transX);
                    cylinderTransform.setTranslation(cylinderTranslation);
                    cylinderGroup.setTransform(cylinderTransform);

                    // Rotates in (0, 0, 0)
                    propellerRotation.rotZ(axisRadians);
                    propellerGroup.setTransform(propellerRotation);

                    // Rotates in (165, 0, 0)
                    crankarmRotation.rotZ(-crankarmRadians);
                    crankarmTemp.mul(crankarmRotation, crankarmCenterTrans);
                    crankarmTransform.mul(crankarmCenterTransInv, crankarmTemp);
                    crankarmGroup.setTransform(crankarmTransform);

                    wakeupOn(activeFramesOrPost);
                }
            } 
        }// End of while
    }

    final class FXCanvas3DSquareImage extends FXCanvas3DImage {

        private FXCanvas3DSquareImage(GraphicsConfigTemplate3D gCT) {
            super(gCT);
        }

        // Set bounds only if bounds are of square size
        @Override
        public void setBounds(int x, int y, int width, int height) {
            if (width == height)
                super.setBounds(x, y, width, height);
        }
    }

}