
package com.interactivemesh.j3d.testspace.jfx.screenflight;

import java.awt.BorderLayout;
import java.awt.Dimension;

import java.awt.event.MouseListener;
import java.awt.event.MouseEvent;

import java.io.FileNotFoundException;

import java.net.URL;

import java.util.Map;

import javax.media.j3d.Alpha;
import javax.media.j3d.Background;
import javax.media.j3d.BoundingSphere;
import javax.media.j3d.Bounds;
import javax.media.j3d.BranchGroup;
import javax.media.j3d.Canvas3D;
import javax.media.j3d.DirectionalLight;
import javax.media.j3d.GraphicsConfigTemplate3D;
import javax.media.j3d.Group;
import javax.media.j3d.Locale;
import javax.media.j3d.OrderedGroup;
import javax.media.j3d.PhysicalBody;
import javax.media.j3d.PhysicalEnvironment;
import javax.media.j3d.RotationInterpolator;
import javax.media.j3d.SceneGraphObject;
import javax.media.j3d.Shape3D;
import javax.media.j3d.PickInfo;
import javax.media.j3d.Transform3D;
import javax.media.j3d.TransformGroup;
import javax.media.j3d.View;
import javax.media.j3d.ViewPlatform;
import javax.media.j3d.VirtualUniverse;

import javax.swing.JPanel;
import javax.swing.SwingUtilities;

import javax.vecmath.AxisAngle4d;
import javax.vecmath.Point3d;
import javax.vecmath.Vector3d;

import com.sun.j3d.utils.pickfast.PickCanvas;

// X3DModelImporter
import com.interactivemesh.j3d.interchange.ext3d.XImportException;
import com.interactivemesh.j3d.interchange.ext3d.XModelReader;

// FXCanvas3D API 3.0, see http://www.interactivemesh.org/testspace/j3dmeetsjfx.html
import com.interactivemesh.j3d.community.gui.FXCanvas3DSB;
import com.interactivemesh.j3d.community.gui.FXCanvas3DRepainter;

// OrbitBehaviorInterim API, see http://www.interactivemesh.org/testspace/orbitbehavior.html
import com.interactivemesh.j3d.community.utils.navigation.orbit.OrbitBehaviorInterim;

// JavaFX
import javafx.async.RunnableFuture;

/**
 * ScreenFlightUniverse.java
 *
 * Version: 3.0
 * Date: 2010/09/19
 *
 * Copyright (c) 2009-2010
 * August Lammersdorf, InteractiveMesh e.K.
 * Kolomanstrasse 2a, 85737 Ismaning
 * Germany / Munich Area
 * www.InteractiveMesh.com/org
 *
 * Please create your own implementation.
 * This source code is provided "AS IS", without warranty of any kind.
 * You are allowed to copy and use all lines you like of this source code
 * without any copyright notice,
 * but you may not modify, compile, or distribute this 'ScreenFlightUniverse.java'.
 *
 */

final class ScreenFlightUniverse implements RunnableFuture {

    static {
        System.out.println("\nScreenFlightUniverse : Copyright (c) 2009-2010 August Lammersdorf, www.InteractiveMesh.com.\n");
    }

    private BoundingSphere      globalBounds        = 	null;

    private View                view                = 	null;
    private Locale              locale              = 	null;

    private Shape3D             propShape3D         =   null;
    private Alpha               propAlpha           =   null;

    private OrbitBehaviorInterim orbitBehInterim    = 	null;

    private PickCanvas          pickCanvas          =   null;

    private BranchGroup         sceneBranch         = 	null;
    private BranchGroup         viewBranch          = 	null;
    private BranchGroup         enviBranch          = 	null;

    ScreenFlightUniverse() {
        // Transparent 3D scene background
//        System.setProperty("j3d.transparentOffScreen", "true");
    }

    // JavaFX Interface RunnableFuture
    @Override
    public void run() {
        initUniverse();
    }

    // Creates and returns the lightweight 3D canvas
    FXCanvas3DSB createFXCanvas3D(FXCanvas3DRepainter repainter, JPanel parent,
                                  boolean fullscreen) {

        FXCanvas3DSB fxCanvas3D = null;

        try {
            GraphicsConfigTemplate3D gCT = new GraphicsConfigTemplate3D();

            fxCanvas3D = new FXCanvas3DSB(gCT);
            fxCanvas3D.setRepainter(repainter);
        }
        catch (Exception e) {
            System.out.println("ScreenFlightUniverse: FXCanvas3D failed !!");
            e.printStackTrace();
            System.exit(0);
        }

        // Transparent component background
        parent.setOpaque(false);
        fxCanvas3D.setOpaque(false);

        // Due to Java 3D's implementation of off-screen rendering:
        // 1. Set size
        // 2. Provide a parent
        // 3. Get the heavyweight off-screen Canvas3D and add this to the view object

        // 1. Size
        Dimension dim = parent.getSize();
        if (dim.width < 1 || dim.height < 1) {
            dim.width = 100;
            dim.height = 100;
            parent.setSize(dim);
        }
        parent.setPreferredSize(dim);
        fxCanvas3D.setPreferredSize(dim);
        fxCanvas3D.setSize(dim);

        // 2. Parent
        // BorderLayout
        parent.setLayout(new BorderLayout());
        parent.add(fxCanvas3D, BorderLayout.CENTER);

        // 3. Heavyweight off-screen Canvas3D of the lightweight FXCanvas3D
        Canvas3D offCanvas3D = fxCanvas3D.getOffscreenCanvas3D();
        if (offCanvas3D != null) {
            // View renders into the off-screen Canvas3D
            view.addCanvas3D(offCanvas3D);
            
            // PickCanvas operates on the not visible off-screen Canvas3D
    	    pickCanvas = new PickCanvas(offCanvas3D, sceneBranch);
    	    pickCanvas.setMode(PickInfo.PICK_GEOMETRY);
    	    pickCanvas.setFlags(PickInfo.NODE);
    	    pickCanvas.setTolerance(4.0f);
        }
        else {
            System.out.println("ScreenFlightUniverse: Off-screen Canvas3D = null !!");
            System.exit(0);
        }

        // MouseAdapter for picking
        MouseListener mouseAdapter = new MouseListener() {
            public void mouseClicked(MouseEvent event) {
                int clicks = event.getClickCount();
                if (clicks == 2 && SwingUtilities.isLeftMouseButton(event)) {
                    pickCanvas.setShapeLocation(event.getX(), event.getY());
                    PickInfo pickInfo = pickCanvas.pickClosest();
                    if (pickInfo != null) {
                        Shape3D pickedShape3D = (Shape3D)pickInfo.getNode();
                        if (pickedShape3D == propShape3D) {
                            startStopPropeller(propAlpha.isPaused());
                        }
                    }
                }
            }
            public void mouseEntered(MouseEvent e) {}
            public void mouseExited(MouseEvent e) {}
            public void mousePressed(MouseEvent e) {}
            public void mouseReleased(MouseEvent e) {}
        };

        fxCanvas3D.addMouseListener(mouseAdapter);

        // Navigator
        setupNavigator(fxCanvas3D, fullscreen);

        return fxCanvas3D;
    }

    //
    // Universe interaction
    //

    boolean isPropellerOn() {
        return !propAlpha.isPaused();
    }
    void startStopPropeller(boolean start) {
        if (start && propAlpha.isPaused())
            propAlpha.resume();
        else if (!start && !propAlpha.isPaused())
            propAlpha.pause();
    }

    void closeUniverse() {
        view.removeAllCanvas3Ds();
        view.attachViewPlatform(null);
        locale.getVirtualUniverse().removeAllLocales();
    }

    //
    // Create universe
    //
    private void initUniverse() {

        createUniverse();
        createScene();

        setLive();
    }

    // Setup navigation
    private void setupNavigator(JPanel component, boolean fullscreen) {

        orbitBehInterim.setAWTComponent(component);

        if (!fullscreen)
            orbitBehInterim.setTranslateEnable(false);

        double sceneRadius = 1.0f;

        Bounds bounds = sceneBranch.getBounds();
        BoundingSphere sphereBounds = null;

        if (bounds.isEmpty()) {
            sphereBounds = new BoundingSphere();
        }
        else {
            if (bounds instanceof BoundingSphere)
                sphereBounds = (BoundingSphere)bounds;
            else
                sphereBounds = new BoundingSphere(bounds);
        }

        sceneRadius = sphereBounds.getRadius();

        orbitBehInterim.setTransFactors(sceneRadius/4.0f, sceneRadius/4.0f);
        orbitBehInterim.setZoomFactor(sceneRadius/4.0f);
        orbitBehInterim.setRotFactors(0.5f, 0.5f);

	orbitBehInterim.setClippingBounds(sphereBounds);

        orbitBehInterim.goHome();
    }

    private void createScene() {

        // p51_mustang.obj / MustangModelJ3D-AllShapes.x3d  
        //Point3d lowerM = new Point3d(-6.68434, -1.96063, -8.49168);
        //Point3d upperM = new Point3d(7.43119, 2.53839, 8.49168);
        
        XModelReader x3dImporter = new XModelReader();
        try {
            x3dImporter.read(this.getClass().getResource("resources/MustangModelJ3D-AllShapes.x3d"));
        }
        catch (XImportException e) {
            e.printStackTrace();
            return;
        }
        
        Map<String, SceneGraphObject> namedSGOs = x3dImporter.getNamedSGOs();   

        // Due to transparent canopy
        OrderedGroup orderedGroup = new OrderedGroup();
        Group mustangGroup = new Group();

        mustangGroup.addChild((Shape3D)namedSGOs.get("Exhaust"));
        mustangGroup.addChild((Shape3D)namedSGOs.get("Fuselage"));
        mustangGroup.addChild((Shape3D)namedSGOs.get("Wings"));
        mustangGroup.addChild((Shape3D)namedSGOs.get("CanopyFrame"));

        orderedGroup.addChild(mustangGroup);        
        orderedGroup.addChild((Shape3D)namedSGOs.get("CanopyGlass")); 
        
        propShape3D = (Shape3D)namedSGOs.get("Propeller");

        TransformGroup propRotTG = new TransformGroup();
        propRotTG.setCapability(TransformGroup.ALLOW_TRANSFORM_WRITE);

        propAlpha = new Alpha();
        propAlpha.setIncreasingAlphaDuration(2000);
        propAlpha.setLoopCount(-1);

        RotationInterpolator propInterpolator = new RotationInterpolator(propAlpha, propRotTG);
        propInterpolator.setSchedulingBounds(globalBounds);
        Transform3D yToYAxis = new Transform3D();
        yToYAxis.setRotation(new AxisAngle4d(0, 0, 1, -Math.PI/2.0));
        yToYAxis.setTranslation(new Vector3d(0, 0.26319, 0));
        propInterpolator.setTransformAxis(yToYAxis);

        propRotTG.addChild(propShape3D);
        mustangGroup.addChild(propRotTG);

        sceneBranch.addChild(propInterpolator);
        sceneBranch.addChild(orderedGroup);
    }

    // Set live
    private void setLive() {
	sceneBranch.compile();
        locale.addBranchGraph(sceneBranch);
        locale.addBranchGraph(viewBranch);
        locale.addBranchGraph(enviBranch);
    }

    // Base world
    private void createUniverse() {

        // Bounds
    	globalBounds = new BoundingSphere();
        globalBounds.setRadius(Double.MAX_VALUE);

        //
        // Viewing
        //
        view = new View();
        view.setPhysicalBody(new PhysicalBody());
        view.setPhysicalEnvironment(new PhysicalEnvironment());

        //
        // SuperStructure
        //
        VirtualUniverse vu = new VirtualUniverse();
        locale = new Locale(vu);

        //
        // BranchGraphs
        //
        sceneBranch = new BranchGroup();
        viewBranch = new BranchGroup();
        enviBranch = new BranchGroup();

        // ViewBranch

        TransformGroup viewTG = new TransformGroup();
        viewTG.setCapability(TransformGroup.ALLOW_TRANSFORM_WRITE);

        ViewPlatform vp = new ViewPlatform();
        view.attachViewPlatform(vp);

        orbitBehInterim = new OrbitBehaviorInterim(OrbitBehaviorInterim.REVERSE_ALL);
        orbitBehInterim.setViewingTransformGroup(viewTG);
        orbitBehInterim.setVpView(view);
        orbitBehInterim.setSchedulingBounds(globalBounds);
        orbitBehInterim.setClippingEnabled(true);

        Transform3D homeTransform = new Transform3D();
        homeTransform.setTranslation(new Vector3d(0.0, 0.0, 30.0));
        orbitBehInterim.setHomeTransform(homeTransform);
        orbitBehInterim.setHomeRotationCenter(new Point3d(0.0, 0.0, 0.0));

        DirectionalLight headLight = new DirectionalLight();
        headLight.setInfluencingBounds(globalBounds);

        viewTG.addChild(vp);
        viewTG.addChild(orbitBehInterim);
        viewTG.addChild(headLight);

        viewBranch.addChild(viewTG);

        // EnviBranch

        Background bg = new Background();
        bg.setApplicationBounds(globalBounds);
        bg.setColor(0, 1, 1);

        enviBranch.addChild(bg);
    }
}
