/*
 * LICENSE
 *
 * TuxInTheBoxMain.fx is a derived work of VideoCube.fx
 *
 * Redistribution and use are permitted according to the following license notice.
 *
 * Version: 4.0
 * Date: 2010/09/20
 *
 * Author:
 * August Lammersdorf, InteractiveMesh e.K.
 * Kolomanstrasse 2a, 85737 Ismaning
 * Germany / Munich Area
 * www.InteractiveMesh.com/org
 *
*/
/*
 * VideoCube.fx
 *
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER
 * Copyright 2008 Sun Microsystems, Inc. All rights reserved. Use is subject to license terms.
 *
 * This file is available and licensed under the following license:
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 *   * Redistributions of source code must retain the above copyright notice,
 *     this list of conditions and the following disclaimer.
 *
 *   * Redistributions in binary form must reproduce the above copyright notice,
 *     this list of conditions and the following disclaimer in the documentation
 *     and/or other materials provided with the distribution.
 *
 *   * Neither the name of Sun Microsystems nor the names of its contributors
 *     may be used to endorse or promote products derived from this software
 *     without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR
 * ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 * ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package com.interactivemesh.j3d.testspace.jfx.canvascube;

import com.javafx.preview.control.Menu;
import com.javafx.preview.control.MenuItem;
import com.javafx.preview.control.PopupMenu;
import com.javafx.preview.control.RadioMenuItem;

import java.lang.System;

import javafx.animation.KeyFrame;
import javafx.animation.Timeline;

import javafx.geometry.HPos;
import javafx.geometry.Insets;
import javafx.geometry.VPos;

import javafx.scene.Cursor;
import javafx.scene.CustomNode;
import javafx.scene.Group;
import javafx.scene.Scene;

import javafx.scene.control.Label;
import javafx.scene.control.Separator;
import javafx.scene.control.Slider;
import javafx.scene.control.ToggleGroup;
import javafx.scene.control.Tooltip;

import javafx.scene.effect.PerspectiveTransform;

import javafx.scene.image.Image;
import javafx.scene.image.ImageView;

import javafx.scene.input.MouseButton;
import javafx.scene.input.MouseEvent;

import javafx.scene.layout.HBox;
import javafx.scene.layout.LayoutInfo;
import javafx.scene.layout.Tile;
import javafx.scene.layout.VBox;

import javafx.scene.paint.Color;

import javafx.scene.shape.Polygon;
import javafx.scene.shape.Rectangle;

import javafx.scene.text.Font;
import javafx.scene.text.FontWeight;

import javafx.stage.Screen;
import javafx.stage.Stage;

import javafx.util.Math;

// TuxInTheBoxMain.fx

// TODO : How to check for application versus applet ??
// API : getArgument("javafx.applet") will return null if not running as an applet
// API : This is an experimental facility, that may be changed in future versions.
var isApplication: Boolean = (FX.getArgument("javafx.applet") == null);

// Frames per second
def updatesPerSecond = 2;
var elapsedFrames: Integer = 50;
var frameCounter: Integer;
var startTimePaint: Number = System.nanoTime();
var endTimePaint: Number;
var frameDuration: Number;
var fpsPaint: Integer;

// Design

def textFG = Color.WHITE;
def backgroundColor = Color.color(0.0, 0.4, 0.8);
def lineColor = Color.WHITE;

def buttonColor = Color.WHITE;
def hoverColor = Color.color(1.0, 0.7, 0.0);   // orange;
def pressColor = Color.color(1.0, 0.2, 0.0);   // dark orange;
def selectColor = Color.color(0.0, 1.0, 0.6);   // neon green

def screenHeight: Number = Screen.primary.bounds.height;
def screenDPI: Number = Screen.primary.dpi;

// screenHeight >= 1200
var textFontSize = 18;
var titleFontSize = 34;
var sceneHeight = 900;
var border = 50;

// screenHeight  < 1024
if (screenHeight < 1024) {
    textFontSize = 14;
    titleFontSize = 26;
    border = 30;
    sceneHeight = 620;
}
// 1024 <= screenHeight < 1200
else if (screenHeight < 1200) {
    textFontSize = 16;
    titleFontSize = 30;
    border = 40;
    sceneHeight = 800;
}

// Bug RT-9312, since 1.3.1
// Logical fonts (Dialog, Serif, etc.) overruns, although text doesn't exceed the available space.

def titleFont = Font.font("Amble Cn", FontWeight.REGULAR, titleFontSize);
def textFont = Font.font("Amble Cn", FontWeight.BOLD, textFontSize);
def menuFont = Font.font("Amble Cn", FontWeight.BOLD, textFontSize);
def tipFont = Font { size: textFontSize};

def controlSpace = 10;
def vertSpace = 3;

// JavaFX box face rectangles
var faces: Face[];

var textureImages: Image[];
var currTextureIndex: Number = 0;

// JavaFX box projection: parallel or perspective
var perspective: Boolean = false;
var switchToPerspective: Boolean = false;

// Box rotation transformation
def rotXMatrix = Matrix3N{};
def rotYMatrix = Matrix3N{};
def rotMatrix = Matrix3N{};
def transformMatrix = Matrix3N{};

def defaultViewpoint = [ 0.70710677,  0.0,        -0.70710677,
                        -0.29883623,  0.90630776, -0.29883623,
                         0.6408563,   0.42261827,  0.6408563  ];

def faceLayoutInfo: LayoutInfo = LayoutInfo{width: 300 height: 300}

// Center box in scene rectangle
var dx: Number;
var dy: Number;

// Size of scene rectangle
def sceneRectWidth: Number = bind stage.scene.width - 2*border on replace {

    dx = sceneRectWidth/2;

    resizeFaces.playFromStart();
};
def sceneRectHeight: Number = bind stage.scene.height - 3*border on replace {

    dy = sceneRectHeight/2;

    resizeFaces.playFromStart();
};

// Runs only once in case of width and height are changed !?
def resizeFaces = Timeline {
    repeatCount: 1
    keyFrames: KeyFrame {
        //canSkip: false;
        time: 1ms
        action: function() {

            perspective = switchToPerspective;
            
            var faceSize: Number;
            if (perspective) {
                faceSize = (Math.min(sceneRectWidth, sceneRectHeight) + 2*border)/(2.0 * 1.0);
            }
            else {
                faceSize = (Math.min(sceneRectWidth, sceneRectHeight) + 2*border)/(2.0 * Math.sqrt(3));
            }

            setupBox(faceSize);
            
            universeFX.fxCanvas3DMVControl.setFXCanvas3DMVSize(faceSize, faceSize);

            universeFX.setFaceTexture(currTextureIndex);

            faceLayoutInfo.width = faceSize;
            faceLayoutInfo.height = faceSize;
        }
    }
}

// Box face
class Face extends CustomNode {

    var name: String;
    var number: Integer;
    var isAdded: Boolean;

    var ul3d: Tuple3N; 
    var ur3d: Tuple3N; 
    var lr3d: Tuple3N; 
    var ll3d: Tuple3N; 

    var isVisible: Boolean = false;

    // Show or hide face
    // Stop 3D rendering if not visible 
    var show: Boolean = false on replace {
        if (show and not isVisible) {
            isVisible = true;
            faceCanvas3Dfx.setRendererRunning(true);
        }
        else if (not show and isVisible) {
            isVisible = false;
            faceCanvas3Dfx.setRendererRunning(false);
        }
    };

    var faceGroup: Group;
    var faceLabel: Label;
    var faceCanvas3Dfx: FXCancas3DComp;

    override function create() {

        def pt = PerspectiveTransform {
            ulx: bind dx + ul3d.x
            uly: bind dy + ul3d.y
            urx: bind dx + ur3d.x
            ury: bind dy + ur3d.y
            lrx: bind dx + lr3d.x
            lry: bind dy + lr3d.y
            llx: bind dx + ll3d.x
            lly: bind dy + ll3d.y
        }

        Group {
            visible: bind isVisible;
            content: [
                // Outline of PerspectiveTransform for receiving mouse events
                Polygon {
                    fill: Color.TRANSPARENT;

                    points: bind [
                        pt.ulx, pt.uly,
                        pt.urx, pt.ury,
                        pt.lrx, pt.lry,
                        pt.llx, pt.lly
                    ]

                    onMousePressed: function (event: MouseEvent) {
                        if (event.button != MouseButton.PRIMARY or isAdded)
                            return;
                        rotation.pause();
                        faceGroup.visible = false;
                        faceGroup.effect = null;
                        jCanvas3DComps[number].initFXCanvas3D(universeFX.getFXCanvas3DMV(number));
                        rotation.play();
                        isAdded = true;
                    }
                },                                
                // Face content: rectangle, text 
                faceGroup = Group {                    
                    effect: pt
                    content: [
                        Rectangle { // cube has gaps at edges
                            width: bind faceLayoutInfo.width
                            height: bind faceLayoutInfo.height
                            fill:  lineColor
                        },
                        faceLabel = Label {
                            translateX: bind (faceLayoutInfo.width - faceLabel.layoutBounds.width)/2
                            translateY: bind (faceLayoutInfo.height-faceLabel.layoutBounds.height)/2
                            text: "Click me! Drag me!"
                            font: textFont
                            textFill: backgroundColor
                        }
                    ]
                },
                // Face content: 3D view panel incl. white line border
                Group {
                    effect: pt
                    content: faceCanvas3Dfx
                }
            ]
        }
    }
}

// FXCanvas3D per box face
def jCanvas3DComps = [FXCancas3DComp{}, FXCancas3DComp{}, FXCancas3DComp{},
                      FXCancas3DComp{}, FXCancas3DComp{}, FXCancas3DComp{}];
                     
var universeFX: TuxInTheBoxUniverseFX = TuxInTheBoxUniverseFX {
    // Callback of AsyncOperation
    initUniverse: function(universe: TuxInTheBoxUniverse): Void {

        universeFX.fxCanvas3DMVControl = universe.getFXC3dMVControl();
        universeFX.fxCanvas3DMVControl.setRepainter(universeFX);

        // Initial box size for parallel projection
        def size = sceneHeight/(2 * Math.sqrt(3));

        setupBox(size);
        universeFX.fxCanvas3DMVControl.setFXCanvas3DMVSize(size, size);

        faceLayoutInfo.width = size;
        faceLayoutInfo.height = size;

        for (face in faces) {
            face.faceCanvas3Dfx.layoutInfo = faceLayoutInfo;
        }

        textureImages = universeFX.getFaceImages();
        for (i in [0..5]) {
            textureRadioList[i].imageR = textureImages[i];
        }

        setViewpoint("Home");

        // Show box
        rotation.play();

        // Show frame
        stage.visible = true;
    }

    fieldOfView: bind (fovSlider.value) as Integer;
}

// Box points
// f:front, b:back | u:upper, l:lower | l:left, r:right
def ful3d = Tuple3N {}
def fur3d = Tuple3N {}
def flr3d = Tuple3N {}
def fll3d = Tuple3N {}

def bul3d = Tuple3N {}
def bur3d = Tuple3N {}
def blr3d = Tuple3N {}
def bll3d = Tuple3N {}

// Transformed box points
def ful3dt = Tuple3N {}
def fur3dt = Tuple3N {}
def flr3dt = Tuple3N {}
def fll3dt = Tuple3N {}

def bul3dt = Tuple3N {}
def bur3dt = Tuple3N {}
def blr3dt = Tuple3N {}
def bll3dt = Tuple3N {}

// Adjust box size according to scene rectangle size
// y upside down ! due to 2D coordinate system
function setupBox(e: Number) {

    ful3d.x = -e; ful3d.y = -e; ful3d.z = e;
    fur3d.x =  e; fur3d.y = -e; fur3d.z = e;
    flr3d.x =  e; flr3d.y =  e; flr3d.z = e;
    fll3d.x = -e; fll3d.y =  e; fll3d.z = e;

    bul3d.x = -e; bul3d.y = -e; bul3d.z = -e;
    bur3d.x =  e; bur3d.y = -e; bur3d.z = -e;
    blr3d.x =  e; blr3d.y =  e; blr3d.z = -e;
    bll3d.x = -e; bll3d.y =  e; bll3d.z = -e;

    transformMatrix.setupPerspectiveView(Math.toRadians(45), 2*e);
}

// Box faces
def faceFront = Face {
    number: 0
    ul3d: ful3dt   ur3d: fur3dt  lr3d: flr3dt   ll3d: fll3dt   // +Z, front face
    faceCanvas3Dfx: jCanvas3DComps[0]
    name: "front"
}
insert faceFront into faces;

def faceBack = Face {
    number: 1
    ul3d: bur3dt   ur3d: bul3dt   lr3d: bll3dt   ll3d: blr3dt   // -Z, back face
    faceCanvas3Dfx: jCanvas3DComps[1]
    name: "back"
}
insert faceBack into faces;

def faceRight = Face {
    number: 2
    ul3d: fur3dt   ur3d: bur3dt   lr3d: blr3dt   ll3d: flr3dt   // +X, right face
    faceCanvas3Dfx: jCanvas3DComps[2]
    name: "right"
}
insert faceRight into faces;

def faceLeft = Face {
    number: 3
    ul3d: bul3dt   ur3d: ful3dt   lr3d: fll3dt   ll3d: bll3dt   // -X, left face
    faceCanvas3Dfx: jCanvas3DComps[3]
    name: "left"
}
insert faceLeft into faces;

def faceTop = Face {
    number: 4
    ul3d: bul3dt   ur3d: bur3dt   lr3d: fur3dt   ll3d: ful3dt   // +Y, top face
    faceCanvas3Dfx: jCanvas3DComps[4]
    name: "top"
}
insert faceTop into faces;

def faceBottom = Face {
    number: 5
    ul3d: fll3dt   ur3d: flr3dt   lr3d: blr3dt   ll3d: bll3dt   // -Y, bottom face
    faceCanvas3Dfx: jCanvas3DComps[5]
    name: "bottom"
}
insert faceBottom into faces;

// Box rotation animation
def rotation = Timeline {
    repeatCount: Timeline.INDEFINITE
    keyFrames: KeyFrame {
        canSkip: true;
        time: 10ms                          // <= 100 frames per second
        action: function() {

            // Apply current rotation
            transformMatrix.multiplyLeft(rotMatrix);

            // Transform box
            if (perspective) {
                transformMatrix.transformPersp(ful3d, ful3dt);
                transformMatrix.transformPersp(fur3d, fur3dt);
                transformMatrix.transformPersp(flr3d, flr3dt);
                transformMatrix.transformPersp(fll3d, fll3dt);

                transformMatrix.transformPersp(bul3d, bul3dt);
                transformMatrix.transformPersp(bur3d, bur3dt);
                transformMatrix.transformPersp(blr3d, blr3dt);
                transformMatrix.transformPersp(bll3d, bll3dt);
            }
            else {
                transformMatrix.transform(ful3d, ful3dt);
                transformMatrix.transform(fur3d, fur3dt);
                transformMatrix.transform(flr3d, flr3dt);
                transformMatrix.transform(fll3d, fll3dt);

                transformMatrix.transform(bul3d, bul3dt);
                transformMatrix.transform(bur3d, bur3dt);
                transformMatrix.transform(blr3d, blr3dt);
                transformMatrix.transform(bll3d, bll3dt);
            }

            // Hidden face removal
            backfaceCulling();

            //
            // Frames per second
            //

            frameCounter++;

            if (frameCounter >= elapsedFrames) {

                endTimePaint = System.nanoTime();
                frameDuration = (endTimePaint-startTimePaint)/frameCounter;
                startTimePaint = endTimePaint;

                fpsPaint = (1000000000 / frameDuration) as Integer;

                // Reset
                frameCounter = 0;
                elapsedFrames = Math.max(1, ((fpsPaint + 0.5) / updatesPerSecond)) as Integer;
            }

        }
    }
}

function backfaceCulling(): Void {
    faceFront.show = (normalZ(ful3dt, fur3dt, fll3dt) > 0);
    faceBack.show = (normalZ(bur3dt, bul3dt, blr3dt) > 0);
    faceRight.show = (normalZ(flr3dt, fur3dt, bur3dt) > 0);
    faceLeft.show = (normalZ(bul3dt, ful3dt, bll3dt) > 0);
    faceTop.show = (normalZ(ful3dt, bul3dt, fur3dt) > 0);
    faceBottom.show = (normalZ(fll3dt, flr3dt, bll3dt) > 0);
}

function normalZ(v0: Tuple3N, v1: Tuple3N, v3: Tuple3N): Number {

    def V01 = Tuple3N{x: v1.x - v0.x, y: v1.y - v0.y, z: v1.z - v0.z};
    def V03 = Tuple3N{x: v3.x - v0.x, y: v3.y - v0.y, z: v3.z - v0.z};

    return (V01.x * V03.y  -  V01.y * V03.x);
}

function setViewpoint(vp: String) {

    rotation.pause();

    rotMatrix.setIdentity();
    
    // Home 
    if (vp.equalsIgnoreCase("Home")) {
        transformMatrix.set(defaultViewpoint);
    }
    // Front face
    else if (vp.equalsIgnoreCase("Front")) {
        transformMatrix.setIdentity();
        // transformMatrix.rotationY(Math.toRadians(-90.0));
    }
    // Top face
    else if (vp.equalsIgnoreCase("Top")) {
        transformMatrix.rotationX(Math.toRadians(-90.0));
    }
    // Bottom face
    else if (vp.equalsIgnoreCase("Bot")) {
        transformMatrix.rotationX(Math.toRadians(90.0));
    }

    rotation.play();
}

//
// Controls
//

// JavaFX box projection mode

def projectionGroup = RadioLabelGroup {
    selectionChanged: function(selectedRadio: RadioLabel) {
        if (selectedRadio == perspectiveRadio and not perspective) {
            toggleProjectionMode()
        }
        else if (selectedRadio == parallelRadio and perspective) {
            toggleProjectionMode()
        }
    }
}
def perspectiveRadio = RadioLabel {
    text: "Perspective"
    tooltip: TuxTooltip { text: "Perspective projection" }
}
def parallelRadio = RadioLabel {
    text: "Parallel"
    tooltip: TuxTooltip { text: "Parallel projection" }
}

projectionGroup.radioList = [parallelRadio, perspectiveRadio];
projectionGroup.setSelection(parallelRadio);

function toggleProjectionMode() {
    rotation.pause();
    switchToPerspective = not perspective;
    resizeFaces.playFromStart();
    rotation.play();
}

// Viewpoints

def homeButton = ButtonLabel {
    text: "Home"
    tooltip: TuxTooltip { text: "Home Viewpoint" }
    action: function(): Void { setViewpoint("Home") }
};
def frontButton = ButtonLabel {
    text: "Front"
    tooltip: TuxTooltip { text: "Front Viewpoint" }
    action: function(): Void { setViewpoint("Front") }
};
def topButton = ButtonLabel {
    text: "Top"
    tooltip: TuxTooltip { text: "Top Viewpoint" }
    action: function(): Void { setViewpoint("Top") }
};
def bottomButton = ButtonLabel {
    text: "Bottom"
    tooltip: TuxTooltip { text: "Bottom Viewpoint" }
    action: function(): Void { setViewpoint("Bot") }
};

// Frames per second : JavaFX

def fpsLabel = BaseLabel {
    text: "F P S"
    tooltip: TuxTooltip { text: "Frames per second" }
}
def fpsValueLabel = BaseLabel {
    text: bind "{fpsPaint}"
    tooltip: TuxTooltip { text: "Frames per second" }
}

// Frames per second : Java 3D

def fps3DLabel = BaseLabel {
    text: "F P S"
    tooltip: TuxTooltip { text: "Frames per second" }
}
def fpsValue3DLabel = BaseLabel {
    text: bind "{universeFX.fpsPaint}"
    tooltip: TuxTooltip { text: "Frames per second" }
}

// Texture selection

def textureGroup = ImageRadioGroup {
    selectionChanged: function(selectedRadio: ImageRadioButton) {
        currTextureIndex = selectedRadio.index;
        universeFX.setFaceTexture(currTextureIndex);
    }
}

var textureRadioList: ImageRadioButton[];

for (i in [0..5]) {
    insert ImageRadioButton {index: i} into textureRadioList;
}

textureGroup.radioList = textureRadioList;
textureGroup.setSelection(textureRadioList[0]);

// Field of View

def fov = Tile {
    columns: 2
    content: [
        BaseLabel { text: "F O V :" tooltip: TuxTooltip { text: "Field of view [10\u00B0, 120\u00B0]" }},
        BaseLabel { text: bind "{(fovSlider.value) as Integer} \u00B0" }
    ]
    layoutInfo: LayoutInfo {hfill: true vfill: true margin: Insets {left: 30, right: 30}};
}

def fovSlider = Slider {
    min: 10 
    max: 120 
    value: 45
    blockIncrement: 1
    layoutInfo: LayoutInfo { hfill: true vfill: true };
}

// JavaFX Controls Layout

def javaFXControls = VBox {
    var tile: Tile;
    var lInfo = LayoutInfo {width: controlSpace*2 height: bind tile.height};
    
    spacing: controlSpace*2
    content: [
        BaseLabel {
            text: "JavaFX"
            font: titleFont
        },
//      Separator { layoutInfo: LayoutInfo {height: controlSpace*2 } },
        HBox {            
            content: [
                // Projection mode
                tile = Tile {
                    hgap: 0 vgap: vertSpace columns: 1
                    content: [
                        parallelRadio, perspectiveRadio
                    ]
                },
                Separator {vertical: true layoutInfo: lInfo},
                // VantagePoints
                Tile {
                    hgap: vertSpace vgap: vertSpace columns: 2
                    content: [ homeButton, topButton, frontButton, bottomButton ]
                },
                Separator {vertical: true layoutInfo: lInfo},
                // F P S : JavaFX
                Tile {
                    hgap: 0 vgap: vertSpace columns: 1
                    content: [ fpsLabel, fpsValueLabel ]
                }
            ]
        }
    ]
}

// Java 3D Controls Layout

def java3DLabel = BaseLabel {
    text: "Java 3D"
    font: titleFont
}

var java3DLabelVBox: Group;
java3DLabelVBox = Group {
    translateX: bind java3DControls.layoutBounds.width - java3DLabel.layoutBounds.width;
    content: java3DLabel
}

def java3DControls = VBox {
    var tile: Tile;
    var lInfo = LayoutInfo {width: controlSpace*2 height: bind tile.height};

    spacing: controlSpace*2
    content: [
        java3DLabelVBox,
//      Separator {layoutInfo: LayoutInfo {height: controlSpace*2 hfill: true margin: Insets {left: 0, right: 0}}},
        HBox {
            content: [
                // F P S : Java 3D
                tile = Tile {
                    hgap: 0 vgap: vertSpace columns: 1
                    content: [ fps3DLabel, fpsValue3DLabel ]
                },
                Separator {vertical: true layoutInfo: lInfo},
                // Texture
                HBox {
                    translateY: 4
                    spacing: controlSpace/2
                    content: textureRadioList
                },
                Separator {vertical: true layoutInfo: lInfo},
                // FOV
                Tile {
                    hgap: 0 vgap: vertSpace columns: 1 
                    content: [ fov, fovSlider ]
                }
            ]
        }
    ]
}

def popupMenu: PopupMenu = PopupMenu {

    def projRadioGroup = ToggleGroup{};
    var parRadio: PopupRadioMenuItem;
    var perRadio: PopupRadioMenuItem;

    var menuItem: PopupSubMenu
    layoutInfo: LayoutInfo {height: bind menuItem.layoutBounds.height * (if (isApplication) 4 else 3) }
    
    items: [
        menuItem = PopupSubMenu {
            text: "Select projection"
            items: [
                parRadio = PopupRadioMenuItem {
                    text: "Parallel" 
                    toggleGroup: projRadioGroup
                    action: function() {
                        projectionGroup.setSelection(parallelRadio);
                        toggleProjectionMode();
                    }
                }
                perRadio = PopupRadioMenuItem {
                    text: "Perspective" 
                    toggleGroup: projRadioGroup
                    action: function() {
                        projectionGroup.setSelection(perspectiveRadio);
                        toggleProjectionMode();
                    }
                }
            ]
        },
        Separator {},
        PopupSubMenu {
            text: "Select viewpoint"
            items: [
                PopupMenuItem { text: "Home" action: function() { setViewpoint("Home") } }
                PopupMenuItem { text: "Front" action: function() { setViewpoint("Front") } }
                PopupMenuItem { text: "Top" action: function() { setViewpoint("Top") } }
                PopupMenuItem { text: "Bottom" action: function() { setViewpoint("Bot") } }
            ]
        }
    ]

    override var onShowing = function() {
        if (perspective) {
            projRadioGroup.selectedToggle = perRadio
        }
        else {
            projRadioGroup.selectedToggle = parRadio
        }
    }

    override var onAction = function(item :MenuItem):Void { hide() }

    visible: false
}
// Complete popupMenu
if (isApplication) {
    insert Separator{} into popupMenu.items;
    insert PopupMenuItem {
        text: "Exit application"
        action: function() { stage.close() }
    } into popupMenu.items;
}

// Headline
def headLine = BaseLabel {
    text: "Java 3D meets JavaFX"
    font: titleFont
};

// Frame / Stage
def stage: Stage = Stage {
    title: "InteractiveMesh : FXTuxInTheBox"
    visible: false
    resizable: true
    onClose: function() {
        if (rotation != null) 
            rotation.stop();
        universeFX.closeUniverse();
    }
    scene: Scene {
        stylesheets: ["{__DIR__}caspian-desktop-tuxinthebox.css"]

        var popupPanel: Rectangle;
        var dragPanel: Rectangle;
        
        fill: backgroundColor
        // Start size
        width: sceneHeight * 1.4
        height: sceneHeight + border

        content: [
            // Context menu
            popupMenu,

            // Background node for popup menu
            popupPanel = Rectangle {
                width: bind stage.scene.width
                height: bind stage.scene.height
                fill: backgroundColor
                onMousePressed: function(event) {
                    if (popupMenu.showing)
                        popupMenu.hide();
                }
                onMouseClicked: function(event) {
                    if (event.button == MouseButton.SECONDARY) {
                        if (isApplication) {
                            popupMenu.show(popupPanel, event.screenX+5, event.screenY);
                        }
                        else { // Applet workaround
                            popupMenu.show(popupPanel, HPos.LEFT, VPos.TOP, event.x, event.y);
                        }
                    }
                }
            },

            // Headline
            Group {
                layoutX: border + controlSpace
                layoutY: bind border - headLine.layoutBounds.height * 0.5
                content: headLine
            },

            // Line border, width = 3
            Rectangle {
                layoutX: border - 3
                layoutY: 2 * border - 3
                width: bind sceneRectWidth + 6
                height: bind sceneRectHeight + 6
                fill: Color.WHITE
            },

            // Mouse navigator: rotation
            dragPanel = Rectangle {
                layoutX: border
                layoutY: 2 * border
                width: bind sceneRectWidth
                height: bind sceneRectHeight
                fill: backgroundColor

                var isDragging = false;
                
                onMousePressed: function(event) {
                    if (event.button == MouseButton.PRIMARY) {
                        dragPanel.cursor = Cursor.MOVE;
                        isDragging = true;
                    }
                }
                onMouseReleased: function(event) {
                    if (isDragging) {
                        dragPanel.cursor = Cursor.DEFAULT;
                        isDragging = false;
                    }
                }
                onMouseDragged: function(event) {
                    if (not isDragging) return;
                    
                    def dragX = event.dragX;
                    def dragY = event.dragY;

                    def dX = if (-5 < dragX and dragX < 5) then 0 else dragX * 0.0001;
                    def dY = if (-5 < dragY and dragY < 5) then 0 else dragY * 0.0001;

                    rotXMatrix.rotationX(-dY);
                    rotYMatrix.rotationY(dX);
                    rotMatrix.multiply(rotYMatrix, rotXMatrix);
                }
            },
            
            // Controls : JavaFX / Java 3D
            Group {
                layoutX: border + controlSpace*2
                layoutY: bind border*2 + sceneRectHeight -controlSpace*2 - javaFXControls.layoutBounds.height
                content: javaFXControls
            },
            Group {
                layoutX: bind border + sceneRectWidth - controlSpace*2 - java3DControls.layoutBounds.width
                layoutY: bind border*2 + sceneRectHeight -controlSpace*2 - javaFXControls.layoutBounds.height
                content: java3DControls
            },
            
            // 3D Box
            Group {
                layoutX: border
                layoutY: 2 * border
                // Faces
                content: faces
            }
        ]
    }
}

//
// Start
//
// JavaTaskBase
universeFX.start();

//
// Control classes
//

class PopupSubMenu extends Menu {
    override var font = menuFont
}
class PopupMenuItem extends MenuItem {
    override var font = menuFont
}
class PopupRadioMenuItem extends RadioMenuItem {
    override var font = menuFont
}

class TuxTooltip extends Tooltip {
    override var font = tipFont
}

class BaseLabel extends Label {
    override var font = textFont;
    override var textFill = buttonColor;
    override var hpos = HPos.CENTER;
    var selected: Boolean = false;
    override var onMouseEntered = function(event: MouseEvent) {
        tooltip.activate();
    };
    override var onMouseExited = function(event: MouseEvent) {
        tooltip.deactivate();
    };
    override var onMousePressed = function(event: MouseEvent) {
        tooltip.deactivate();
    };
}

class ButtonLabel extends BaseLabel {

    var action: function(): Void;

    init {
        blocksMouse = true
    }

    override var onMouseEntered = function(event: MouseEvent) {
        if (not pressed) {
            textFill = hoverColor;
        }
        tooltip.activate();
    };
    override var onMouseExited = function(event: MouseEvent) {
        if (not pressed) {
            textFill = buttonColor;
        }
        tooltip.deactivate();
    };
    override var onMousePressed = function(event: MouseEvent) {
        textFill = pressColor;
        tooltip.deactivate();
    };
    override var onMouseReleased = function(event: MouseEvent) {
        if (disabled)
            return;

        if (hover) {
            action();
            textFill = hoverColor;
        }
        else {
            textFill = buttonColor;
        }
    };
}

class RadioLabel extends BaseLabel {

    var group: RadioLabelGroup;

    var selectAction: function(): Void;

    init {
        blocksMouse = true
    }

    override var onMouseEntered = function(event: MouseEvent) {
        if (not pressed and not selected) {
            textFill = hoverColor;
        }
        tooltip.activate();
    };
    override var onMouseExited = function(event: MouseEvent) {
        if (not pressed and not selected) {
            textFill = buttonColor;
        }
        tooltip.deactivate();
    };
    override var onMousePressed = function(event: MouseEvent) {
        if (not selected) {
            pressed = true;
            textFill = pressColor;
        }
        tooltip.deactivate();
    };
    override var onMouseReleased = function(event: MouseEvent) {
        if (selected or disabled)
            return;
        pressed = false;
        if (hover) {
            group.select(this);
        }
        else {
            textFill = buttonColor;
        }
    };
}

class RadioLabelGroup {

    var radioList: RadioLabel[] on replace {
        for (radio in radioList) {
            radio.group = this;
        }
    };

    var selectionChanged: function(radio: RadioLabel);

    function setSelection(selectedRadio: RadioLabel) {
        for (radio in radioList) {
            radio.selected = false;
            radio.textFill = buttonColor;
        }

        selectedRadio.selected = true;
        selectedRadio.textFill = selectColor;
    }

    function select(selectedRadio: RadioLabel) {
        setSelection(selectedRadio);
        selectionChanged(selectedRadio);
    }
}
//
class ImageRadioButton extends CustomNode {

    var radioGroup: ImageRadioGroup;
    var selected: Boolean;
    var index: Integer = -1;

    var defaultColor: Color;
    
    var rect: Rectangle;
    var imageR: Image;
    
    init {
        defaultColor = textFG;
    }

    function selectMe() {
        radioGroup.select(this);
    }

    override function create(): Group {
        return Group {
            content: [
                rect = Rectangle {
                    opacity: 0.0
                    blocksMouse: true;
                    width: 32+8
                    height: 32+8

                    override var onMouseEntered = function(event: MouseEvent) {
                        if (not selected and not pressed) {
                            opacity = 1.0;
                            fill = hoverColor;
                        }
                    };
                    override var onMouseExited = function(event: MouseEvent) {
                        if (not selected and not pressed) {
                            opacity = 0.0;
                        }
                    };
                    override var onMousePressed = function(event: MouseEvent) {
                        if (not selected) {
                            fill = pressColor;
                        }
                    };
                    override var onMouseReleased = function(event: MouseEvent) {
                        if (selected)
                            return;

                        if (hover) {
                            selectMe();
                        }
                        else {
                            opacity = 0.0;
                            fill = defaultColor;
                         }
                    };
                },
                ImageView {
                    image: bind imageR
                    layoutX: 4
                    layoutY: 4
                    cache: true
                }
            ]                                    
        }
    };
}
class ImageRadioGroup {

    var radioList: ImageRadioButton[] on replace {
        for (radio in radioList) {
            radio.radioGroup = this;
        }
    };

    var selectionChanged: function(radio: ImageRadioButton);

    function setSelection(selectedRadio: ImageRadioButton) {
        for (radio in radioList) {
            radio.selected = false;
            radio.rect.opacity = 0.0;
        }

        selectedRadio.selected = true;
        selectedRadio.rect.fill = selectColor;
        selectedRadio.rect.opacity = 1.0;
    }

    function select(selectedRadio: ImageRadioButton) {
        setSelection(selectedRadio);
        selectionChanged(selectedRadio);
    }
}
