package com.interactivemesh.j3d.community.gui;

/*
 * LICENSE
 * 
 * JCanvas3DOBAbstract.java 
 * 
 * is a derived work of the Java 3D utility class "com.sun.j3d.exp.swing.JCanvas3D.java".
 * 
 * Redistribution and use are permitted according to the following license notice.
 * 
 * Version: 1.0
 * Date: 2009/03/09
 * 
 * Author:
 * August Lammersdorf, InteractiveMesh e.K.
 * Kolomanstrasse 2a, 85737 Ismaning
 * Germany / Munich Area
 * www.InteractiveMesh.com/org
 * 
*/

/*
 * $RCSfile: com.sun.j3d.exp.swing.JCanvas3D.java,v $
 *
 * Copyright (c) 2007 Sun Microsystems, Inc. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * - Redistribution of source code must retain the above copyright
 *   notice, this list of conditions and the following disclaimer.
 *
 * - Redistribution in binary form must reproduce the above copyright
 *   notice, this list of conditions and the following disclaimer in
 *   the documentation and/or other materials provided with the
 *   distribution.
 *
 * Neither the name of Sun Microsystems, Inc. or the names of
 * contributors may be used to endorse or promote products derived
 * from this software without specific prior written permission.
 *
 * This software is provided "AS IS," without a warranty of any
 * kind. ALL EXPRESS OR IMPLIED CONDITIONS, REPRESENTATIONS AND
 * WARRANTIES, INCLUDING ANY IMPLIED WARRANTY OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE OR NON-INFRINGEMENT, ARE HEREBY
 * EXCLUDED. SUN MICROSYSTEMS, INC. ("SUN") AND ITS LICENSORS SHALL
 * NOT BE LIABLE FOR ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF
 * USING, MODIFYING OR DISTRIBUTING THIS SOFTWARE OR ITS
 * DERIVATIVES. IN NO EVENT WILL SUN OR ITS LICENSORS BE LIABLE FOR
 * ANY LOST REVENUE, PROFIT OR DATA, OR FOR DIRECT, INDIRECT, SPECIAL,
 * CONSEQUENTIAL, INCIDENTAL OR PUNITIVE DAMAGES, HOWEVER CAUSED AND
 * REGARDLESS OF THE THEORY OF LIABILITY, ARISING OUT OF THE USE OF OR
 * INABILITY TO USE THIS SOFTWARE, EVEN IF SUN HAS BEEN ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGES.
 *
 * You acknowledge that this software is not designed, licensed or
 * intended for use in the design, construction, operation or
 * maintenance of any nuclear facility.
 *
 * $Revision: 1.10 $
 * $Date: 2007/04/11 02:08:56 $
 * $State: Exp $
 * 
 *  @author: Frederic 'pepe' Barachant
 *
 */

import java.awt.Container;
import java.awt.Dimension;
import java.awt.Graphics;
import java.awt.GraphicsConfigTemplate;
import java.awt.GraphicsConfiguration;
import java.awt.GraphicsDevice;
import java.awt.GraphicsEnvironment;
import java.awt.Rectangle;

import java.awt.event.ComponentEvent;
import java.awt.event.FocusEvent;
import java.awt.event.InputMethodEvent;
import java.awt.event.KeyEvent;
import java.awt.event.MouseEvent;
import java.awt.event.MouseWheelEvent;

import java.awt.image.BufferedImage;
import java.awt.image.DataBufferInt;

import java.util.concurrent.locks.Condition;
import java.util.concurrent.locks.ReentrantLock;

import javax.media.j3d.Canvas3D;
import javax.media.j3d.GraphicsConfigTemplate3D;
import javax.media.j3d.ImageComponent2D;
import javax.media.j3d.RestrictedAccessException;
import javax.media.j3d.Screen3D;

import javax.swing.JPanel;
import javax.swing.SwingUtilities;

import javax.swing.event.AncestorEvent;
import javax.swing.event.AncestorListener;

import com.sun.j3d.exp.swing.impl.AutoOffScreenCanvas3D;

/**
 * <p>
 * This class provides Java 3D rendering into a lightweight component 
 * which can be added to a Swing container.
 * </p> 
 * <p>
 * It allows to run the Java 3D rendering loop in parallel to the 
 * Swing painting loop with synchronized minimal overlapping.  
 * </p> 
 * <p>
 * The following events will be redirected to the underlying heavyweight Canvas3D object:<br>
 * <ul>
 * <li>InputMethodEvent</li>
 * <li>KeyEvent</li>
 * <li>FocusEvent</li>
 * <li>ComponentKeyEvent</li>
 * <li>MouseWheelEvent</li>
 * <li>MouseEvent</li>
 * <li>MouseMotionEvent</li>
 * </ul>
 * </p>
 *
 * Serialization: JCanvas3DOBAbstract does not support serialization. 
 *  
 * @since Java 3D 1.5
 * 
 */
// License see above
abstract public class JCanvas3DOBAbstract extends    JPanel 
                                          implements AncestorListener {

    /** size of a pixel */
    private static double METERS_PER_PIXEL = 0.0254 / 90.0;

    /** the graphics configuration used for this canvas */
    private GraphicsConfiguration graphicsConfig;

    /** The canvas that is linked to the component. */
    private InternalCanvas3D canvas;
    
    //
    // Double Off-screen Buffer - see also InternalCanvas3D
    //
    /**
     * That image of the off-screen buffers which Swing is drawing.
     * @see #paintComponent(Graphics)
     */
    protected BufferedImage paintImage  =   null;
    /**
     * The width of the image which Swing is drawing.
     */
    protected volatile int  imageWidth  =   0;      
    /**
     * The height of the image which Swing is drawing.
     */
    protected volatile int  imageHeight =   0;      
    
    /**
     * Constructs and initializes a new JCanvas3DOBAbstract object that Java 3D
     * can render into. The screen device is obtained from
     * <code>GraphicsEnvironment.getLocalGraphicsEnvironment().getDefaultScreenDevice()</code>,
     * which might not be the one you should use if you are in a multiscreen environment.
     * The JCanvas3DOBAbstract is constructed using the following default parameters:<br>
     * double buffer enable : false<br>
     * stereo enable : false<br>
     */
    protected JCanvas3DOBAbstract() {
        this(null, null);
    }

    /**
     * Constructs and initializes a new JCanvas3DOBAbstract object that Java 3D
     * can render into, using the specified graphics device.
     *
     * @param device the screen graphics device that will be used to construct
     *        a GraphicsConfiguration.
     */
    protected JCanvas3DOBAbstract(GraphicsDevice device) {
        this(null, device);
    }

    /**
     * Constructs and initializes a new JCanvas3DOBAbstract object that Java 3D
     * can render into, using the specified template.
     * The screen device is obtained from
     * <code>GraphicsEnvironment.getLocalGraphicsEnvironment().getDefaultScreenDevice()</code>,
     * which might not be the one you should use if you are
     * in a multiscreen environment.
     *
     * @param template The template that will be used to construct a
     *        GraphicsConfiguration. The stereo and doublebuffer properties
     *        are forced to UNNECESSARY.
     */
    protected JCanvas3DOBAbstract(GraphicsConfigTemplate3D template) {
        this(template, null);
    }

    /**
     * Constructs and initializes a new JCanvas3DOBAbstract object that Java 3D
     * can render into, using the specified template and graphics device.
     *
     * @param template The template that will be used to construct a
     *        GraphicsConfiguration. The stereo and doublebuffer properties
     *        are forced to UNNECESSARY.
     * @param device the screen graphics device that will be used to construct
     *        a GraphicsConfiguration in conjunction with the template.
     */
    protected JCanvas3DOBAbstract(GraphicsConfigTemplate3D template, GraphicsDevice device) {
        
        if (template == null) {
            template = new GraphicsConfigTemplate3D();
        }
        // Force double-buffer and stereo to UNNECESSARY
        template.setStereo(GraphicsConfigTemplate.UNNECESSARY);
        template.setDoubleBuffer(GraphicsConfigTemplate.UNNECESSARY);

        if (device == null) {
            graphicsConfig = GraphicsEnvironment.getLocalGraphicsEnvironment().
                                getDefaultScreenDevice().getBestConfiguration(template);
        }
        else {       
            graphicsConfig = device.getBestConfiguration(template);
        }

        addAncestorListener(this);
        setDoubleBuffered(false);

        // so that key events and such can be received.
        setFocusable(true);
    }
    
    //
    // Interface AncestorListener
    //
    
    /**
     * Called when the source or one of its ancestors is made visible either 
     * by setVisible(true) being called or by its being added to the component hierarchy. 
     * 
     * The method is only called if the source has actually become visible.
     * For this to be true all its parents must be visible and 
     * it must be in a hierarchy rooted at a Window.
     * 
     * @param event {@inheritDoc}
     */
     public void ancestorAdded(AncestorEvent event) {
        Dimension sz = getSize();

        if (sz.width == 0) {
            sz.width = 100;
        }

        if (sz.height == 0) {
            sz.height = 100;
        }

        createOffScreenBuffer(sz.width, sz.height);

        canvas.addNotifyFlag = true; // make it so that i can call addNotify() without being rejected.
        canvas.addNotify();
    }

    /**
     * Called when either the source or one of its ancestors is moved.
     *
     * @param event {@inheritDoc}
     */
    public void ancestorMoved(AncestorEvent event) {
    }

    /**
     * Called when the source or one of its ancestors is made invisible either 
     * by setVisible(false) being called or by its being remove from the component hierarchy. 
     * 
     * The method is only called if the source has actually become invisible. 
     * For this to be true at least one of its parents must by invisible or 
     * it is not in a hierarchy rooted at a Window. 
     *
     * @param event {@inheritDoc}
     */
    public void ancestorRemoved(AncestorEvent event) {
        canvas.removeNotify();
    }

    //
    // Java 3D specific methods
    //
            
    /**
     * Returns the off-screen heavyweight canvas of this lightweight component.
     * 
     * No canvas is created if this component has no parent, that is, 
     * was not added to a container.   
     * 
     * @return the heavyweight canvas that lies in the deepness of this
     *         Component.
     */
    public Canvas3D getOffscreenCanvas3D() {
        if (canvas == null) {
            createOffScreenBuffer(getWidth(), getHeight());
        }

        return canvas;
    }
    
    /**
     * {@inheritDoc}
     *
     * @param x {@inheritDoc}
     * @param y {@inheritDoc}
     * @param width {@inheritDoc}
     * @param height {@inheritDoc}
     */
    @Override
    public void setBounds(int x, int y, int width, int height) {

        super.setBounds(x, y, width, height);

        createOffScreenBuffer(width, height);           
    }
    
    
    /**
     * Creates new off-screen buffers of the given size. This method is called 
     * internally whenever this panel is added to a parent or is resized. 
     * <p>
     * Subclasses should call and/or override this method according to its
     * individual needs. In case of overriding calling
     * <code>super.createOffScreenBuffer(canvasWidth, canvasHeight)</code> 
     * has to be the last thing to do.</p>
     * @param width the width of the off-screen buffers to create
     * @param height the height of the off-screen buffers to create
     */
    protected void createOffScreenBuffer(int width, int height) {        
        createCanvas(width, height);       
    }
    
    /**
     * Callback used to allow an overriding subclass to execute individual code
     * when new off-screen buffers were created. 
     * <p>
     * This method is called internally by the event-dispatching thread (EDT)
     * and should not be called by applications. 
     * </p>
     */
    protected void offScreenBufferCreated () {       
    }
    
    /**
     * Callback used to allow an overriding subclass to execute individual code
     * when the off-screen buffers were copied.
     * <p>
     * This method is called internally by the event-dispatching thread (EDT)
     * and should not be called by applications. 
     * </p>
     */
    protected void offScreenBufferCopied() {        
    }

    /**
     * Subclasses which overrides <code>paintComponent(Graphics g)</code>
     * should check if the off-screen buffer is ready for drawing.
     * 
     * @return true if off-screen buffer can be drawn
     */
    protected final boolean isReadyForDrawing() {
        return (paintImage != null && canvas.isRendererRunning());
    }    

    /**
     * Creates a heavyweight canvas and initializes it, or changes the
     * size of the current one if present. Current heavyweight canvas is
     * changed only if size is different from the actual one. No canvas is
     * created if this component has no parent, that is, was not added to a
     * container.
     *
     * @param width the width of the canvas to create.
     * @param height the height of the canvas to create.
     */
    private void createCanvas(final int width, final int height) {
       
        if (getParent() == null) {
            return;
        }
        
        if (canvas == null) {
            canvas = new InternalCanvas3D(graphicsConfig);
        } 
        else {
            if (canvas.getWidth() == width && canvas.getHeight() == height)
                return;
        }

        // Create 'Double OffScreen Buffer'

        if (SwingUtilities.isEventDispatchThread()) {
            canvas.createOffScreenBuffers(graphicsConfig.getBounds(), width, height);
        }
        else {
            SwingUtilities.invokeLater(new Runnable() {
                public void run() {
                    canvas.createOffScreenBuffers(graphicsConfig.getBounds(), width, height);
                }
            });
        }
    }
    
    /**
     * Flips and paints the result of the 3D rendering.
     *
     * @param g {@inheritDoc}
     */
    @Override
    public void paintComponent(Graphics g) {
        
        super.paintComponent(g); // paint background
        
        // To be implemented in subclasses.
        
        // Call super.paintComponent(g) for painting the background
        
        /* Draw & flip offscreen buffer
         
        if (isReadyForDrawing()) {
        
            g.drawImage(paintImage,
                    // destination in g2d: flip lowerY and upperY 
                    // dx1    dy1          dx2         dy2
                        0, imageHeight, imageWidth, 0,    
                    // source: the bottom part of the scene image
                    // sx1    sy1          sx2         sy2
                        0, 0,           imageWidth, imageHeight, null);
        }
        
        */
    }
    
    //
    // Override Component's processXXX
    //
    /**
     * Redirects event to canvas and to superclass.
     *
     * @param e {@inheritDoc}
     */
    @Override
    protected void processComponentKeyEvent(KeyEvent e) {
        super.processComponentKeyEvent(e);

        Object src = e.getSource();
        e.setSource(canvas);
        canvas.processComponentEvent(e);
        e.setSource(src);
    }

    /**
     * Redirects event to canvas and to superclass.
     *
     * @param e {@inheritDoc}
     */
    @Override
    protected void processFocusEvent(FocusEvent e) {
        super.processFocusEvent(e);

        Object src = e.getSource();
        e.setSource(canvas);
        canvas.processFocusEvent(e);
        e.setSource(src);
    }

    /**
     * Redirects event to canvas and to superclass.
     *
     * @param e {@inheritDoc}
     */
    @Override
    protected void processInputMethodEvent(InputMethodEvent e) {
        super.processInputMethodEvent(e);

        Object src = e.getSource();
        e.setSource(canvas);
        canvas.processInputMethodEvent(e);
        e.setSource(src);
    }

    /**
     * Redirects event to canvas and to superclass.
     *
     * @param e {@inheritDoc}
     */
    @Override
    protected void processKeyEvent(KeyEvent e) {
        super.processKeyEvent(e);

        Object src = e.getSource();
        e.setSource(canvas);
        canvas.processKeyEvent(e);
        e.setSource(src);
    }

    /**
     * Redirects event to canvas and to superclass.
     *
     * @param e {@inheritDoc}
     */
    @Override
    protected void processMouseEvent(MouseEvent e) {
        super.processMouseEvent(e);

        Object src = e.getSource();
        e.setSource(canvas);
        canvas.processMouseEvent(e);
        e.setSource(src);
    }

    /**
     * Redirects event to canvas and to superclass.
     *
     * @param e {@inheritDoc}
     */
    @Override
    protected void processMouseMotionEvent(MouseEvent e) {
        super.processMouseMotionEvent(e);

        Object src = e.getSource();
        e.setSource(canvas);
        canvas.processMouseMotionEvent(e);
        e.setSource(src);
    }

    /**
     * Redirects event to canvas and to superclass.
     *
     * @param e {@inheritDoc}
     */
    @Override
    protected void processMouseWheelEvent(MouseWheelEvent e) {
        super.processMouseWheelEvent(e);

        Object src = e.getSource();
        e.setSource(canvas);
        canvas.processMouseWheelEvent(e);
        e.setSource(src);
    }

    /**
     * This class is the internal Canvas3D that is used and sent to
     * Java 3D. It is remote controlled through JCanvas3DOBAbstract and is modified to be
     * able to tell the lightweight component when refreshes are needed.
     */
    private final class InternalCanvas3D extends    Canvas3D 
                                         implements AutoOffScreenCanvas3D {
        // Double Offscreen Buffer
        
        private ReentrantLock   imageAccessLock     =   null;
        private Condition       imageCopyCondition  =   null;
        private Condition       imageResizeCondition =  null;
        private boolean         isImageCopied       =   false;
        private boolean         isRendererLocked    =   false;
        
        private BufferedImage   renderImage         =   null;

        private int[]           paintIntBuffer      =   null;
        private int[]           renderIntBuffer     =   null;
        
        private Runnable        repainter           =   null;

        /**
         * Flag used to sort a call to addnotify() from user and
         * from the lightweight component. Lightweight component calls
         * addNotify() so that the rendering begins and uses normal routines,
         * but this is a method that user must not call.
         */
        private boolean addNotifyFlag = false;
        
        // Don't wait longer than 200 ms for repainting
        private static final int LIMIT = 200000000;
        
        /**
         * Creates a new instance of JCanvas3DOBAbstract.
         * 
         * @param graphicsConfiguration The graphics configuration to be used.
         * @param lwCanvas the lightweight canvas that is linked to that
         *        heavyweight canvas.
         */
        private InternalCanvas3D(GraphicsConfiguration graphicsConfiguration) {
            super(graphicsConfiguration, true);
            
            // Sync image access
            imageAccessLock      = new ReentrantLock();
            imageCopyCondition   = imageAccessLock.newCondition();
            imageResizeCondition = imageAccessLock.newCondition();
            
            repainter = new Repainter();
        }
        
        // To be called on the EDT only !!!!!!!!!!!!!!
        /**
         * Creates an offscreen buffer to be attached to the heavyweight
         * buffer. Buffer is created 'byreference'
         *
         * @param width the width of the buffer.
         * @param height the height of the buffer.
         */
        private void createOffScreenBuffers(Rectangle screenRect, int width, int height) {

            // Now the J3D-Renderer thread can't enter the repaint code block in postSwap
            this.stopRenderer(); 

            // Compute physical dimensions of the screen
            // Fix to Issue : 433 - JCanvas3D crashed when using jogl pipe.
            int screenWidth = (int)screenRect.getWidth();
            int screenHeight = (int)screenRect.getHeight();
            Screen3D screen3D = this.getScreen3D();
            screen3D.setSize(screenWidth, screenHeight);
            screen3D.setPhysicalScreenWidth(((double) screenWidth) * METERS_PER_PIXEL);
            screen3D.setPhysicalScreenHeight(((double) screenHeight) * METERS_PER_PIXEL);
            
            if (renderImage != null)
                renderImage.flush();
            if (paintImage != null)
                paintImage.flush();
            
            // OffScreenBuffer: byReference & yUp
            renderImage = new BufferedImage(width, height, BufferedImage.TYPE_INT_ARGB);
            ImageComponent2D image = new ImageComponent2D(ImageComponent2D.FORMAT_RGBA, renderImage, true, true);
            image.setCapability(ImageComponent2D.ALLOW_IMAGE_READ);
            image.setCapability(ImageComponent2D.ALLOW_IMAGE_WRITE);
            
            paintImage = new BufferedImage(width, height, BufferedImage.TYPE_INT_ARGB);

            paintIntBuffer  = ((DataBufferInt)paintImage.getRaster().getDataBuffer()).getData();
            renderIntBuffer = ((DataBufferInt)renderImage.getRaster().getDataBuffer()).getData();

            imageWidth  = width;
            imageHeight = height;
            
            // Callback: new off-screen buffer created
            offScreenBufferCreated();
            
            try {
                imageAccessLock.lock();
                
                // Setting offscreen buffer requires that the J3D-Renderer thread isn't blocked
                // As we are on the EDT, 'this.run()' will not release the J3D-Renderer
                // So, it's done here. We will wait until this has happened.
                while (isRendererLocked) {
                    
                     isImageCopied = true;
                     imageCopyCondition.signal();
                     
                     try {
                        imageResizeCondition.await();
                     }
                     catch (InterruptedException e) {
                     }
                 }
                
                // Offscreen rendering might occur even if the renderer is stopped. 
                // For that reason, we are waiting for a hypothetical offscreen render 
                // to finish before setting offscreen rendering.
                // Otherwise, rendering will stop with an exception.
                this.waitForOffScreenRendering();

                try {
                    // EDT will wait (sleep) until exchange of offscreen buffer is done 
                    this.setOffScreenBuffer(image);                
                }
                catch (RestrictedAccessException e) {
                    // TODO Try again, riks of endless loop ?
                    createOffScreenBuffers(screenRect, width, height);
                    System.out.println("Repeat : createOffScreenBuffer(screenRect, width, height) !!");
                }
            }
            finally  {
                imageAccessLock.unlock();
            }

            this.startRenderer();
        }
        
        /**
         * {@inheritDoc}
         */
        @Override
        public void addNotify() {
            if (addNotifyFlag) {
                addNotifyFlag = false;
                super.addNotify();                
            } 
            else {
                throw new UnsupportedOperationException("CHANGE ME");
            }
        }

        /**
         * Normally, returns the parent of that component. As the
         * canvas ought to never be added to any component, it has no parent.
         * Java 3D expects it to have a parent for some operations, so we in
         * fact cheat it by returning the parent of the lightweight component.
         *
         * @return the parent of the lightweight component, if any. Returns
         *         null if  it has no parent.
         */
        @Override
        public Container getParent() {
            return JCanvas3DOBAbstract.this.getParent();
        }

        /**
         * Retrieves the buffer from canvas, if possible, and
         * calls/notifies component to be repainted, if necessary.
         */
        @Override
        public void postSwap() {
            
            if (isRendererRunning()) {   
                try {
                    imageAccessLock.lock();
          
                    isImageCopied = false;
                    
                    // Call 'System.arraycopy()' and 'repaint()' on EDT                    
                    SwingUtilities.invokeLater(repainter); 
                    
                    // Wait until image is copied
                    while (!isImageCopied) { 
                        isRendererLocked = true;
                        
                        try {                           
                            imageCopyCondition.awaitNanos(LIMIT);   // Don't wait for ever
                            isImageCopied = true;                   // and release yourself
                        }
                        catch (InterruptedException e) {
                        }
                    }
                    
                    isRendererLocked = false;
                    imageResizeCondition.signal();
                }
                finally  {                      
                   imageAccessLock.unlock();
                }                
            } 
        }
        
        // Repainter
        /** 
         * Copies the 3D rendering image into the 2D painting image 
         * while the J3D-Renderer thread is waiting
         * and repaints the lightweight panel.
         */
        private class Repainter implements Runnable {
            public void run() {                
                try {      
                    imageAccessLock.lock();
                    
                    // Copy 3D rendering image into 2D painting image
                    System.arraycopy(renderIntBuffer, 0, paintIntBuffer, 0, renderIntBuffer.length);
                    
                    // Notify the waiting J3D-Renderer thread.
                    isImageCopied = true;
                    imageCopyCondition.signal();        
                }
                finally {
                    imageAccessLock.unlock();
                }

                // Callback: off-screen buffer copied
                offScreenBufferCopied();                 
                    
                JCanvas3DOBAbstract.this.repaint();
                                               
                // paintImmediately : scene runs less smooth
                //JCanvas3DOBAbstract.this.paintImmediately(0, 0, imageWidth, imageHeight);
            }
        }
        //
        // Override Component's processXXX
        //
        /**
         * Overriden so that the JComponent can access it.
         *
         * @param e {@inheritDoc}
         */
        @Override
        protected void processComponentEvent(ComponentEvent e) {
            super.processComponentEvent(e);
        }

        /**
         * Overriden so that the JComponent can access it.
         *
         * @param e {@inheritDoc}
         */
        @Override
        protected void processFocusEvent(FocusEvent e) {
            super.processFocusEvent(e);
        }

        /**
         * Overriden so that the JComponent can access it.
         *
         * @param e {@inheritDoc}
         */
        @Override
        protected void processInputMethodEvent(InputMethodEvent e) {
            super.processInputMethodEvent(e);
        }

        /**
         * Overriden so that the JComponent can access it.
         *
         * @param e {@inheritDoc}
         */
        @Override
        protected void processKeyEvent(KeyEvent e) {
            super.processKeyEvent(e);
        }

        /**
         * Overriden so that the JComponent can access it.
         *
         * @param e {@inheritDoc}
         */
        @Override
        protected void processMouseEvent(MouseEvent e) {
            super.processMouseEvent(e);
        }

        /**
         * Overriden so that the JComponent can access it.
         *
         * @param e {@inheritDoc}
         */
        @Override
        protected void processMouseMotionEvent(MouseEvent e) {
            super.processMouseMotionEvent(e);
        }

        /**
         * Overriden so that the JComponent can access it.
         *
         * @param e {@inheritDoc}
         */
        @Override
        protected void processMouseWheelEvent(MouseWheelEvent e) {
            super.processMouseWheelEvent(e);
        }
    }
}
