package org.interactivemesh.pivot.wtk.j3d;

/*
* LICENSE
* 
* PCanvas3D API 2.0
* 
* This API is a derived work from the Java 3D utility class "com.sun.j3d.exp.swing.JCanvas3D.java"
* 
* and includes the file
* 
* - "org.interactivemesh.pivot.wtk.j3d.PCanvas3D.java"
* 
* Redistribution and use are permitted according to the following license notice.
* 
* Version: 2.0
* Date: 2011/01/15
* 
* Author:
* August Lammersdorf, InteractiveMesh e.K.
* Kolomanstrasse 2a, 85737 Ismaning
* Germany / Munich Area
* www.InteractiveMesh.com/org
* 
*/
/*
* $RCSfile: com.sun.j3d.exp.swing.JCanvas3D.java,v $
*
* Copyright (c) 2007 Sun Microsystems, Inc. All rights reserved.
*
* Redistribution and use in source and binary forms, with or without
* modification, are permitted provided that the following conditions
* are met:
*
* - Redistribution of source code must retain the above copyright
*   notice, this list of conditions and the following disclaimer.
*
* - Redistribution in binary form must reproduce the above copyright
*   notice, this list of conditions and the following disclaimer in
*   the documentation and/or other materials provided with the
*   distribution.
*
* Neither the name of Sun Microsystems, Inc. or the names of
* contributors may be used to endorse or promote products derived
* from this software without specific prior written permission.
*
* This software is provided "AS IS," without a warranty of any
* kind. ALL EXPRESS OR IMPLIED CONDITIONS, REPRESENTATIONS AND
* WARRANTIES, INCLUDING ANY IMPLIED WARRANTY OF MERCHANTABILITY,
* FITNESS FOR A PARTICULAR PURPOSE OR NON-INFRINGEMENT, ARE HEREBY
* EXCLUDED. SUN MICROSYSTEMS, INC. ("SUN") AND ITS LICENSORS SHALL
* NOT BE LIABLE FOR ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF
* USING, MODIFYING OR DISTRIBUTING THIS SOFTWARE OR ITS
* DERIVATIVES. IN NO EVENT WILL SUN OR ITS LICENSORS BE LIABLE FOR
* ANY LOST REVENUE, PROFIT OR DATA, OR FOR DIRECT, INDIRECT, SPECIAL,
* CONSEQUENTIAL, INCIDENTAL OR PUNITIVE DAMAGES, HOWEVER CAUSED AND
* REGARDLESS OF THE THEORY OF LIABILITY, ARISING OUT OF THE USE OF OR
* INABILITY TO USE THIS SOFTWARE, EVEN IF SUN HAS BEEN ADVISED OF THE
* POSSIBILITY OF SUCH DAMAGES.
*
* You acknowledge that this software is not designed, licensed or
* intended for use in the design, construction, operation or
* maintenance of any nuclear facility.
*
* $Revision: 1.10 $
* $Date: 2007/04/11 02:08:56 $
* $State: Exp $
* 
*  @author: Frederic 'pepe' Barachant
*
*/

// Java

import java.awt.AWTEvent;
import java.awt.Color;
import java.awt.EventQueue;
import java.awt.Graphics2D;
import java.awt.GraphicsConfigTemplate;
import java.awt.GraphicsConfiguration;
import java.awt.GraphicsDevice;
import java.awt.GraphicsEnvironment;
import java.awt.Rectangle;
import java.awt.Transparency;

import java.awt.event.ComponentEvent;
import java.awt.event.FocusEvent;
import java.awt.event.HierarchyEvent;
import java.awt.event.HierarchyListener;
import java.awt.event.InputEvent;
import java.awt.event.KeyEvent;
import java.awt.event.MouseEvent;
import java.awt.event.MouseWheelEvent;

import java.awt.image.BufferedImage;

import java.util.concurrent.TimeUnit;
import java.util.concurrent.locks.Condition;
import java.util.concurrent.locks.ReentrantLock;

// Java 3D

import javax.media.j3d.Canvas3D;
import javax.media.j3d.GraphicsConfigTemplate3D;
import javax.media.j3d.ImageComponent2D;
import javax.media.j3d.RestrictedAccessException;
import javax.media.j3d.Screen3D;

import com.sun.j3d.exp.swing.impl.AutoOffScreenCanvas3D;

// Apache Pivot

import org.apache.pivot.wtk.ApplicationContext.DisplayHost;
import org.apache.pivot.wtk.Component;
import org.apache.pivot.wtk.Container;
import org.apache.pivot.wtk.Display;
import org.apache.pivot.wtk.Keyboard;
import org.apache.pivot.wtk.Keyboard.KeyLocation;
import org.apache.pivot.wtk.Mouse;

import org.apache.pivot.wtk.skin.ComponentSkin;

/**
 * A Pivot component that Java 3D can render into.
 * 
 */
public class PCanvas3D extends Component {
    
    // Size of a pixel 
    private static double METERS_PER_PIXEL = 0.0254 / 90.0;
    
    // The graphics configuration used for this canvas 
    private GraphicsConfiguration   graphicsConfig  =   null;

    // The canvas that is linked to the component. 
    private OffscreenCanvas3D       canvas          =   null;
    
    private java.awt.Container      awtParent       =   null;
    
    private long                    wakeupEventMasks =  0;
    // j3d.transparentOffScreen=true
    private boolean                 isTransOffScreen =  false;
    
    //
    // Single Off-screen Buffer
    //
    private BufferedImage       paintImage          =   null;
    private int                 imageWidth          =   0;      
    private int                 imageHeight         =   0;      
    //
    // Lock - Condition
    //
    private ReentrantLock       imageAccessLock     =   null;
    private Condition           imagePaintCondition =   null;
    private volatile boolean    isImageDrawn        =   false;
    
/*
private long startTimePaint = 0;
private long endTimePaint = 0;
private long startTimeDraw = 0;

private long startTimePre = 0;
private long startTimeSwap = 0;
private long endTimeSwap = 0;
private long endTimeWait = 0;
private long startTimeRender = 0;
private long endTimeRender = 0;
*/
    
    /**
     * Constructs and initializes a new PCanvas3D object that Java 3D
     * can render into. The screen device is obtained from
     * <code>GraphicsEnvironment.getLocalGraphicsEnvironment().getDefaultScreenDevice()</code>,
     * which might not be the one you should use if you are in a multiscreen environment.
     * The PCanvas3D is constructed using the following default parameters:<br>
     * double buffer enable : false<br>
     * stereo enable : false<br>
     */
    public PCanvas3D() {
        this(null, null);
    }

    /**
     * Constructs and initializes a new PCanvas3D object that Java 3D
     * can render into, using the specified graphics device.
     *
     * @param device the screen graphics device that will be used to construct
     *        a GraphicsConfiguration.
     */
    public PCanvas3D(GraphicsDevice device) {
        this(null, device);
    }

    /**
     * Constructs and initializes a new PCanvas3D object that Java 3D
     * can render into, using the specified template.
     * The screen device is obtained from
     * <code>GraphicsEnvironment.getLocalGraphicsEnvironment().getDefaultScreenDevice()</code>,
     * which might not be the one you should use if you are
     * in a multiscreen environment.
     *
     * @param template The template that will be used to construct a
     *        GraphicsConfiguration. The stereo and doublebuffer properties
     *        are forced to UNNECESSARY.
     */
    public PCanvas3D(GraphicsConfigTemplate3D template) {
        this(template, null);
    }
    
    /**
     * Constructs and initializes a new FXCanvas3DImage object, 
     * using the specified template and graphics device.
     *
     * @param template The template that will be used to construct a
     *        GraphicsConfiguration. The stereo and doublebuffer properties
     *        are forced to UNNECESSARY.
     * @param device the screen graphics device that will be used to construct
     *        a GraphicsConfiguration in conjunction with the template.
     */
    public PCanvas3D(GraphicsConfigTemplate3D template, GraphicsDevice device) {
        
        if (template == null) {
            template = new GraphicsConfigTemplate3D();
        }
        // Force double-buffer and stereo to UNNECESSARY
        template.setStereo(GraphicsConfigTemplate.UNNECESSARY);
        template.setDoubleBuffer(GraphicsConfigTemplate.UNNECESSARY);

        if (device == null) {
            graphicsConfig = GraphicsEnvironment.getLocalGraphicsEnvironment().
                                getDefaultScreenDevice().getBestConfiguration(template);
        }
        else {       
            graphicsConfig = device.getBestConfiguration(template);
        }

        // Sync image access
        imageAccessLock      = new ReentrantLock();
        imagePaintCondition  = imageAccessLock.newCondition();
        
        // Skin
        PCanvas3DSkin skin = new PCanvas3DSkin();
        setSkin(skin);
        
        // Not secure system property. 'set' and 'get' require signed jars 
        // when used on web in JWS or Applet
        try {
            isTransOffScreen = Boolean.parseBoolean(System.getProperty("j3d.transparentOffScreen"));
        } 
        catch (SecurityException exception) {
        }

    }
    
    /**
     * Sets the Display object for this PCanvas3D object. The top level AWT container 
     * is retrieved from this Diplay object and returned to Java 3D as 'parent' of the
     * underlying heavyweight offscreen canvas.
     * 
     * @param display Display object of the application
     */
    public void setApplicationDisplay(Display display) {
        
        // java.awt.Component
        DisplayHost displayHost = display.getDisplayHost();
        // java.awt.Frame / java.applet.Applet
        awtParent = displayHost.getParent();
        
        // Listen for windowed and fullscreen mode switches
        // DesktopApplicationContext: windowedHostFrame <--> fullScreenHostFrame
        if (awtParent instanceof java.awt.Frame) {
            displayHost.addHierarchyListener(new HierarchyListener() {
                
//              @Override
                public void hierarchyChanged(HierarchyEvent event) {

                    long flags = event.getChangeFlags();
                    
                    if ( (flags & HierarchyEvent.PARENT_CHANGED) != 0 ) {
                        
                        java.awt.Container parent = event.getChangedParent();
                        
                        // Removed
                        if (awtParent == parent) {
                            canvas.removeNotify();
                            awtParent = null;
                        }
                        // Added
                        else {                    
                            awtParent = parent;   
                            if (canvas != null) {
                                canvas.addNotifyFlag = true;
                                canvas.addNotify();
                            }
                        }
                        
                    }
                }
            });
        }
    }
    
    // WakeupOnAWTEvent: COMPONENT_EVENT_MASK, FOCUS_EVENT_MASK, 
    //                   KEY_EVENT_MASK
    //                   MOUSE_EVENT_MASK, MOUSE_MOTION_EVENT_MASK, MOUSE_WHEEL_EVENT_MASK
    /**
     * Sets the AWT event masks for which this PCanvas3D object 
     * will call the corresponding process-methods on the underlying heavyweight canvas.
     * <p>
     * This is required only when Java 3D WakeupOnAWTEvents are specified in subclasses of Behavior. 
     * <p>
     * No event mask is set per default.
     * 
     * @param eventMasks ored AWT event masks: <code>AWTEvent.COMPONENT_EVENT_MASK,
     * AWTEvent.FOCUS_EVENT_MASK, AWTEvent.KEY_EVENT_MASK, AWTEvent.MOUSE_EVENT_MASK,
     * AWTEvent.MOUSE_MOTION_EVENT_MASK, AWTEvent.MOUSE_WHEEL_EVENT_MASK</code>.
     */
    public void enableWakeupOnAWTEvents(long eventMasks) {       
        if ((eventMasks & AWTEvent.COMPONENT_EVENT_MASK) != 0)
            wakeupEventMasks |= AWTEvent.COMPONENT_EVENT_MASK;
        if ((eventMasks & AWTEvent.FOCUS_EVENT_MASK) != 0)
            wakeupEventMasks |= AWTEvent.FOCUS_EVENT_MASK;
        if ((eventMasks & AWTEvent.KEY_EVENT_MASK) != 0)
            wakeupEventMasks |= AWTEvent.KEY_EVENT_MASK;
        if ((eventMasks & AWTEvent.MOUSE_EVENT_MASK) != 0)
            wakeupEventMasks |= AWTEvent.MOUSE_EVENT_MASK;
        if ((eventMasks & AWTEvent.MOUSE_MOTION_EVENT_MASK) != 0)
            wakeupEventMasks |= AWTEvent.MOUSE_MOTION_EVENT_MASK;
        if ((eventMasks & AWTEvent.MOUSE_WHEEL_EVENT_MASK) != 0)
            wakeupEventMasks |= AWTEvent.MOUSE_WHEEL_EVENT_MASK;        
    }
    /**
     * Disables calling event process-methods on the underlying heavyweight canvas
     * for the specified AWT event masks.
     * @param eventMasks ored AWT event masks: <code>AWTEvent.COMPONENT_EVENT_MASK,
     * AWTEvent.FOCUS_EVENT_MASK, AWTEvent.KEY_EVENT_MASK, AWTEvent.MOUSE_EVENT_MASK,
     * AWTEvent.MOUSE_MOTION_EVENT_MASK, AWTEvent.MOUSE_WHEEL_EVENT_MASK</code>
     */
    public void disableWakeupOnAWTEvents(long eventMasks) {
        wakeupEventMasks &= ~eventMasks;
    }
    
    /**
     * Returns the offscreen heavyweight canvas of this lightweight component.
     *
     * @return the heavyweight canvas.
     */
    public Canvas3D getOffscreenCanvas3D() {
        if (this.canvas == null) {
            createOffscreenCanvas3D(getWidth(), getHeight());
        }
        
        return this.canvas;
    }

    /**
     * Creates a heavyweight canvas and initializes it, or changes the
     * size of the current one if present. Current heavyweight canvas is
     * changed only if size is different from the actual one. 
     * 
     * @param width the width of the canvas to create.
     * @param height the height of the canvas to create.
     */
    private void createOffscreenCanvas3D(int width, int height) {
        
        final int w = Math.max(1, width); 
        final int h = Math.max(1, height); 
        
        if (canvas == null) {
            canvas = new OffscreenCanvas3D(this.graphicsConfig);
        } 
        else {
            if (canvas.getWidth() == w && canvas.getHeight() == h)
                return;
        }
        
        // Create 'Single OffScreen Buffer'

        if (EventQueue.isDispatchThread()) {
            canvas.createOffScreenBuffer(graphicsConfig.getBounds(), w, h);
        }
        else {
            EventQueue.invokeLater(new Runnable() { 
                public void run() {
                    canvas.createOffScreenBuffer(graphicsConfig.getBounds(), w, h);
                }
            });
        }
    }
    
    /**
     * This class is the offscreen Canvas3D that is used and sent to
     * Java 3D. It is remote controlled through PCanvas3D and is modified to be
     * able to tell the lightweight component when refreshes are needed.
     */
    private final class OffscreenCanvas3D extends    Canvas3D 
                                          implements AutoOffScreenCanvas3D {

        private Condition   imageResizeCondition    =   null;
        private boolean     isRendererLocked        =   false;
        private boolean     isWatingForResizing     =   false;  
        private boolean     isSwapStarted           =   false;  

        /*
         * Flag used to sort a call to addnotify() from user and
         * from the lightweight component. Lightweight component calls
         * addNotify() so that the rendering begins and uses normal routines,
         * but this is a method that user must not call.
         */
        private boolean addNotifyFlag = false;
        
        // Call repaint on EDT
        private Runnable repaintCall = null;
        
        // Don't wait longer than 200 ms for repainting
        private static final int LIMIT = 200000000;
        
        /**
         * Creates a new instance of OffscreenCanvas3D.
         * 
         * @param graphicsConfiguration The graphics configuration to be used.
         */
        private OffscreenCanvas3D(GraphicsConfiguration graphicsConfiguration) {
            super(graphicsConfiguration, true);
            
            repaintCall = new RepaintCall();
            
            imageResizeCondition = imageAccessLock.newCondition();
        }
        
        // To be called on the EDT only !!!!!!!!!!!!!!
        /**
         * Creates an offscreen buffer to be attached to the heavyweight
         * buffer. Buffer is created 'byReference' and 'yUp'.
         *
         * @param width the width of the buffer.
         * @param height the height of the buffer.
         */
        private void createOffScreenBuffer(Rectangle screenRect, int width, int height) {

            try {
                imageAccessLock.lock();
                
                // 
                this.stopRenderer(); 
                
                isWatingForResizing = true;

                // Setting offscreen buffer requires that the J3D-Renderer thread isn't blocked
                // As we are on the EDT, PCanvas3D can't release the J3D-Renderer
                // So, it's done here. We are waiting until this has happened.
                while (isRendererLocked) {
                    
                    isImageDrawn = true;
                    imagePaintCondition.signal();
                     
                    try {
                        imageResizeCondition.await();
                    }
                    catch (InterruptedException e) {
                    }
                }
                
                isWatingForResizing = false;
                
                // Offscreen rendering might occur even if the renderer is stopped. 
                // For that reason, we are waiting for a hypothetical offscreen render 
                // to finish before setting offscreen rendering.
                // Otherwise, rendering will stop with an exception.
                this.waitForOffScreenRendering();         
                               
                // Compute physical dimensions of the screen
                int screenWidth = (int)screenRect.getWidth();
                int screenHeight = (int)screenRect.getHeight();
                Screen3D screen3D = this.getScreen3D();
                screen3D.setSize(screenWidth, screenHeight);
                screen3D.setPhysicalScreenWidth(((double) screenWidth) * METERS_PER_PIXEL);
                screen3D.setPhysicalScreenHeight(((double) screenHeight) * METERS_PER_PIXEL);
                
                if (paintImage != null)
                    paintImage.flush();
                
                // OffScreenBuffer: byReference & yUp !! 
                paintImage = new BufferedImage(width, height, BufferedImage.TYPE_INT_ARGB);
                ImageComponent2D image = new ImageComponent2D(ImageComponent2D.FORMAT_RGBA, paintImage, true, true);
                image.setCapability(ImageComponent2D.ALLOW_IMAGE_WRITE);
                
                imageWidth  = width;
                imageHeight = height;             

                try {
                    // EDT will wait (sleep) until exchange of offscreen buffer is done 
                    this.setOffScreenBuffer(image);                
                }
                catch (RestrictedAccessException e) {
                    // Try again! Risk of endless loop ?  TODO 
                    createOffScreenBuffer(screenRect, width, height); 
                    System.out.println("Repeat : createOffScreenBuffer(screenRect, width, height) !!");
                }
            }
            finally  {
                imageAccessLock.unlock();
            }

            this.startRenderer();
        }
        
        /*
         * 
         */
        @Override
        public void addNotify() {
            if (addNotifyFlag) {
                addNotifyFlag = false;
                super.addNotify();                
            } 
            else {
                throw new UnsupportedOperationException("User must not call this method !!");
            }
        }

        /**
         * Normally, returns the parent of that component. As the
         * canvas ought to never be added to any component, it has no parent.
         * Java 3D expects it to have a parent for some operations, so we in
         * fact cheat it by returning Pivot's top level AWT container.
         *
         */
        @Override 
        public java.awt.Container getParent() {
            return awtParent;
        }
        
        @Override
        public void preRender() {
//startTimePre = System.nanoTime();     
            if (isSwapStarted) {
                isSwapStarted = false;
                try {
                    imageAccessLock.unlock();                    
                }
                catch (IllegalMonitorStateException e) {                    
                }
            }
        }
        @Override
        public void postRender() {
//startTimeSwap = System.nanoTime(); 
            if (this.isRendererRunning()) {             
                imageAccessLock.lock();              
                isSwapStarted = true;          
            }
        }
        /**
         * Repaint PCanvas3D
         */
        @Override
        public void postSwap() {
//endTimeSwap = System.nanoTime();
            
            if (isSwapStarted) {   
                
                isImageDrawn = false;
                
                EventQueue.invokeLater(repaintCall);
                // PCanvas3D.this.repaint();  TODO How to do repainting best ? 
                
                while (!isImageDrawn) {
                    
                    isRendererLocked = true;
                    
                    try {                           
                        imagePaintCondition.awaitNanos(LIMIT);   // Don't wait for ever
                        isImageDrawn = true;                     // and release yourself
                    }
                    catch (InterruptedException e) {
                    }
//endTimeWait = System.nanoTime();         
                    isRendererLocked = false;
                    if (isWatingForResizing) {
                        imageResizeCondition.signal();
                    }
                }
                
                isSwapStarted = false;
                
                try {
                    imageAccessLock.unlock();                    
                }
                catch (IllegalMonitorStateException e) {                    
                }            
            } 
/* 
endTimeRender = System.nanoTime();
System.out.println(" swap / wait / loop / = " + (endTimeSwap-startTimeSwap) + " / "  + (endTimeWait - endTimeSwap) + " / " + (endTimeRender-startTimeRender));

startTimeRender = endTimeRender;  
 */ 
        }
        
        private final class RepaintCall implements Runnable {
            public void run() {
                PCanvas3D.this.repaint();
            }
        }
        
        // WakeupOnAWTEvent: COMPONENT_EVENT_MASK, FOCUS_EVENT_MASK, 
        //                   KEY_EVENT_MASK
        //                   MOUSE_EVENT_MASK, MOUSE_MOTION_EVENT_MASK, MOUSE_WHEEL_EVENT_MASK

        // Canvas3D / WakeupOnAWTEvent
        // EventCatcher: FocusListener, KeyListener, 
        //               MouseListener, MouseMotionListener, MouseWheelListener
        
        // Canvas3D, all ancestors, Window
        // EventCatcher: ComponentListener
        
        // Canvas3D, all ancestors, Window
        // CanvasViewEventCatcher: ComponentAdapter (componentResized/componentMoved) 

        // Window
        // EventCatcher: WindowListener
        
        //
        // Override Component's processXXX
        //
        /**
         * Overriden so that PCanvas3DSkin can access it.
         *
         * @param e {@inheritDoc}
         */
        @Override
        protected void processComponentEvent(ComponentEvent e) {
            super.processComponentEvent(e);
        }

        /**
         * Overriden so that PCanvas3DSkin can access it.
         *
         * @param e {@inheritDoc}
         */
        @Override
        protected void processFocusEvent(FocusEvent e) {
            super.processFocusEvent(e);
        }

        /**
         * Overriden so that PCanvas3DSkin can access it.
         *
         * @param e {@inheritDoc}
         */
        @Override
        protected void processKeyEvent(KeyEvent e) {
            super.processKeyEvent(e);
        }

        /**
         * Overriden so that PCanvas3DSkin can access it.
         *
         * @param e {@inheritDoc}
         */
        @Override 
        protected void processMouseEvent(MouseEvent e) {
            super.processMouseEvent(e);
        }

        /**
         * Overriden so that PCanvas3DSkin can access it.
         *
         * @param e {@inheritDoc}
         */
        @Override 
        protected void processMouseMotionEvent(MouseEvent e) {
            super.processMouseMotionEvent(e);
        }

        /**
         * Overriden so that PCanvas3DSkin can access it.
         *
         * @param e {@inheritDoc}
         */
        @Override 
        protected void processMouseWheelEvent(MouseWheelEvent e) {
            super.processMouseWheelEvent(e);
        }
    }
    
    //
    // Skin
    //
    
    public final class PCanvas3DSkin extends ComponentSkin {
        
        private Color backgroundColor = null;
        
        public Color getBackgroundColor() {
            return backgroundColor;
        }

        public void setBackgroundColor(Color backgroundColor) {
            this.backgroundColor = backgroundColor;
            repaintComponent();
        }

        // TODO
//      @Override
        public int getPreferredWidth(int width) {
            return getWidth();
        }
//      @Override
        public int getPreferredHeight(int height) {
            return getHeight();
        }
          
//      @Override
        public void layout() {
             // no-op
        }
        
        @Override
        public boolean isFocusable() {
            return true;
        }
        
        @Override
        public boolean isOpaque() { 
             boolean opaque = false;

             if ((backgroundColor != null &&
                  backgroundColor.getTransparency() == Transparency.OPAQUE) ||
                 !isTransOffScreen) {
                 opaque = true;
             }

             return opaque;
        }

        // Create offscreen canvas
        @Override
        public void setSize(int width, int height) {
             super.setSize(width, height);
             createOffscreenCanvas3D(width, height);
        }

        // Paint canvas
//      @Override
        public void paint(Graphics2D g) {             
                    
            Rectangle clip = g.getClipBounds();
            
            if (backgroundColor != null) {
                g.setPaint(backgroundColor);
                g.fillRect(0, 0, clip.width, clip.height);
            }
            
            if (paintImage == null || !canvas.isRendererRunning()) {
                return;
            }

            boolean isLocked = false;
            try {
                // Avoids deadlock when J3D-Renderer thread acquires the lock 
                // in 'postRender' but don't call 'postSwap' due to J3D internal states
                isLocked = imageAccessLock.tryLock(50, TimeUnit.MILLISECONDS);
                
//System.out.println(" paint ClipBounds = " + clip); 
                
//startTimeDraw = System.nanoTime(); 

                // Flip and draw

                if (clip.width != imageWidth || clip.height != imageHeight) {                   
                    // Flip and draw clip area only
                    g.drawImage(paintImage,
                              // destination: Graphics g, flip lowerY and upperY 
                              // dx1    dy1                                  dx2                dy2
                                clip.x, clip.y+clip.height,                  clip.x+clip.width, clip.y,    
                              // source: BufferedImage paintImage
                              // sx1    sy1                                  sx2                sy2
                                clip.x, imageHeight - (clip.y+clip.height),  clip.x+clip.width, imageHeight-clip.y, null);
                }
                else {
                    // Flip and draw entire image
                    g.drawImage(paintImage,
                            // destination: Graphics g, flip lowerY and upperY 
                            // dx1    dy1          dx2         dy2
                                0, imageHeight, imageWidth, 0,    
                            // source: BufferedImage paintImage
                            // sx1    sy1          sx2         sy2
                                0, 0,           imageWidth, imageHeight, null);
                }
                                   
                isImageDrawn = true;
                if (isLocked)
                    imagePaintCondition.signal();
            }
            catch (InterruptedException e) {               
            }
            finally {                      
/*
endTimePaint = System.nanoTime();
long duration = endTimePaint-startTimePaint;
System.out.println(" draw / paint loop / fps = " + (endTimePaint - startTimeDraw) + " / " + duration + " / " + (int)( 1000000000.0 / duration ));
startTimePaint = endTimePaint; 
*/ 
                if (isLocked)
                    imageAccessLock.unlock();
            }         
        }
         
                 
        // Component key events
         
        // KeyEvent(Component source, int id, long when, int modifiers, int keyCode, char keyChar)
        // AWT keyCode = Pivot keyCode 
         
        @Override
        public boolean keyTyped(Component component, char character) {
            boolean consumed = super.keyTyped(component, character);
            if (((wakeupEventMasks & AWTEvent.KEY_EVENT_MASK) != 0) && canvas != null) {
                KeyEvent event = new KeyEvent(canvas, KeyEvent.KEY_TYPED, 
                                           System.currentTimeMillis(), getModifiersEx(), 
                                           KeyEvent.VK_UNDEFINED, character);
                canvas.processKeyEvent(event);
                consumed = true;
            }
            return consumed;
        }

        @Override
        public boolean keyPressed(Component component, int keyCode, KeyLocation keyLocation) {
            boolean consumed = super.keyPressed(component, keyCode, keyLocation);
            if (((wakeupEventMasks & AWTEvent.KEY_EVENT_MASK) != 0) && canvas != null) {
                KeyEvent event = new KeyEvent(canvas, KeyEvent.KEY_PRESSED, 
                                           System.currentTimeMillis(), getModifiersEx(), 
                                           keyCode, KeyEvent.CHAR_UNDEFINED,
                                           getKeyLoc(keyLocation));
                canvas.processKeyEvent(event);
                consumed = true;
            }
            return consumed;
        }

        @Override
        public boolean keyReleased(Component component, int keyCode, KeyLocation keyLocation) {
            boolean consumed = super.keyReleased(component, keyCode, keyLocation);
            if (((wakeupEventMasks & AWTEvent.KEY_EVENT_MASK) != 0) && canvas != null) {
                KeyEvent event = new KeyEvent(canvas, KeyEvent.KEY_RELEASED, 
                                           System.currentTimeMillis(), getModifiersEx(), 
                                           keyCode, KeyEvent.CHAR_UNDEFINED,
                                           getKeyLoc(keyLocation));
                canvas.processKeyEvent(event);
                consumed = true;
            }
            return consumed;
        }

         
        // Component mouse events 
         
        // MouseEvent(Component source, int id, long when, int modifiers, int x, int y, int clickCount, boolean popupTrigger, int button)
         
        @Override
        public boolean mouseMove(Component component, int x, int y) {
            boolean consumed = super.mouseMove(component, x, y);
            if (((wakeupEventMasks & AWTEvent.MOUSE_MOTION_EVENT_MASK) != 0) && canvas != null) {
                int id = MouseEvent.MOUSE_MOVED;
                if (Mouse.isPressed(Mouse.Button.LEFT)  || 
                    Mouse.isPressed(Mouse.Button.RIGHT) ||
                    Mouse.isPressed(Mouse.Button.MIDDLE)   ) {
                    id = MouseEvent.MOUSE_DRAGGED;
                }
                MouseEvent mouseEvent = new MouseEvent(canvas, id, 
                                 System.currentTimeMillis(), getModifiersEx(),  
                                 x, y, 1, false, getButtons());
                canvas.processMouseMotionEvent(mouseEvent);  
                consumed = true;
            }
            return consumed;
        }

        @Override
        public void mouseOver(Component component) {
            super.mouseOver(component);
            if (((wakeupEventMasks & AWTEvent.MOUSE_EVENT_MASK) != 0) && canvas != null) {
                MouseEvent mouseEvent = new MouseEvent(canvas, MouseEvent.MOUSE_ENTERED, 
                                                System.currentTimeMillis(), getModifiersEx(),  
                                                0, 0, 1, false, getButtons());
                canvas.processMouseEvent(mouseEvent);  
            }
        }

        @Override
        public void mouseOut(Component component) {
            super.mouseOut(component);             
            if (((wakeupEventMasks & AWTEvent.MOUSE_EVENT_MASK) != 0) && canvas != null) {
                MouseEvent mouseEvent = new MouseEvent(canvas, MouseEvent.MOUSE_EXITED, 
                                                System.currentTimeMillis(), getModifiersEx(),  
                                                0, 0, 1, false, getButtons());
                canvas.processMouseEvent(mouseEvent);  
            }
        }

        // Component mouse button events
         
        @Override
        public boolean mouseDown(Component component, Mouse.Button button, int x, int y) {
            boolean consumed = super.mouseDown(component, button, x, y);
                                 
            PCanvas3D.this.requestFocus(); // TODO 
             
            if (((wakeupEventMasks & AWTEvent.MOUSE_EVENT_MASK) != 0) && canvas != null) {
                MouseEvent mouseEvent = new MouseEvent(canvas, MouseEvent.MOUSE_PRESSED, 
                                                System.currentTimeMillis(), getModifiersEx(),  
                                                x, y, 1, false, getButtons());
                canvas.processMouseEvent(mouseEvent);
                 consumed = true;
             }
             return consumed;
         }

        @Override
        public boolean mouseUp(Component component, Mouse.Button button, int x, int y) {
            boolean consumed = super.mouseUp(component, button, x, y);
            if (((wakeupEventMasks & AWTEvent.MOUSE_EVENT_MASK) != 0) && canvas != null) {
                MouseEvent mouseEvent = new MouseEvent(canvas, MouseEvent.MOUSE_RELEASED, 
                                                System.currentTimeMillis(), getModifiersEx(),  
                                                x, y, 1, false, getButtons());
                canvas.processMouseEvent(mouseEvent);
                consumed = true;
            }
            return consumed;
        }

        @Override
        public boolean mouseClick(Component component, Mouse.Button button, int x, int y, int count) {
            boolean consumed = super.mouseClick(component, button, x, y, count);
            if (((wakeupEventMasks & AWTEvent.MOUSE_EVENT_MASK) != 0) && canvas != null) {
                MouseEvent mouseEvent = new MouseEvent(canvas, MouseEvent.MOUSE_CLICKED, 
                                                System.currentTimeMillis(), getModifiersEx(),  
                                                x, y, count, false, getButtons());
                canvas.processMouseEvent(mouseEvent);
                consumed = true;
            }
            return consumed;
        }

        // Component mouse wheel events
        // MouseWheelEvent(Component source, int id, long when, int modifiers, int x, int y, int clickCount,
        //                 boolean popupTrigger, int scrollType, int scrollAmount, int wheelRotation)
        @Override
        public boolean mouseWheel(Component component, Mouse.ScrollType scrollType, int scrollAmount,
                                  int wheelRotation, int x, int y) {
            boolean consumed = false;
            if (((wakeupEventMasks & AWTEvent.MOUSE_WHEEL_EVENT_MASK) != 0) && canvas != null) {
                int type = MouseWheelEvent.WHEEL_UNIT_SCROLL;
                if (scrollType == Mouse.ScrollType.BLOCK)
                    type = MouseWheelEvent.WHEEL_BLOCK_SCROLL;
                MouseWheelEvent mouseEvent = new MouseWheelEvent(canvas, MouseEvent.MOUSE_WHEEL,
                         System.currentTimeMillis(), getModifiersEx(), x, y, 1,
                         false, type, scrollAmount, wheelRotation);                
                canvas.processMouseWheelEvent(mouseEvent);       
                consumed = true;
            }
            else {
            	consumed = super.mouseWheel(component, scrollType, scrollAmount, wheelRotation, x, y);
            }
            return consumed;
        }
         
        // Component events
         
        @Override
        public void parentChanged(Component component, Container previousParent) {
             
            Container currParent = component.getParent();
             
            // Removed
            if (currParent == null) {
                if (canvas != null) {
                    canvas.removeNotify();
                }
            }
            // Added
            else if (previousParent == null) {
                 
                createOffscreenCanvas3D(currParent.getWidth(), currParent.getHeight());
                 
                if (canvas != null) {
                    canvas.addNotifyFlag = true;
                    canvas.addNotify();
                }
            }
        }
        
        @Override
        public void visibleChanged(Component component) {
            if (((wakeupEventMasks & AWTEvent.COMPONENT_EVENT_MASK) != 0) && canvas != null) {
                int id = ComponentEvent.COMPONENT_SHOWN;
                if (!component.isVisible()) 
                    id = FocusEvent.COMPONENT_HIDDEN;
                canvas.processComponentEvent(new ComponentEvent(canvas, id));
            }
        }
        
        @Override
        public void locationChanged(Component component, int previousX, int previousY) {
            if (((wakeupEventMasks & AWTEvent.COMPONENT_EVENT_MASK) != 0) && canvas != null)
                canvas.processComponentEvent(new ComponentEvent(canvas, ComponentEvent.COMPONENT_MOVED));
        }
         
        @Override
        public void sizeChanged(Component component, int previousWidth, int previousHeight) {
            if (((wakeupEventMasks & AWTEvent.COMPONENT_EVENT_MASK) != 0) && canvas != null)
                canvas.processComponentEvent(new ComponentEvent(canvas, ComponentEvent.COMPONENT_RESIZED));
        }
         
        // Component state events
         
        @Override
        public void enabledChanged(Component component) {
            // No-op
        }

        @Override
        public void focusedChanged(Component component, Component obverseComponent) {
            super.focusedChanged(component, obverseComponent);
             if (((wakeupEventMasks & AWTEvent.FOCUS_EVENT_MASK) != 0) && canvas != null) {
                 int id = FocusEvent.FOCUS_GAINED;
                 if (!component.isFocused())
                     id = FocusEvent.FOCUS_LOST;             
                 canvas.processFocusEvent(new FocusEvent(canvas, id));
             }
        }
         
        // 
         
        private int getKeyLoc(KeyLocation loc) {
            if (loc == KeyLocation.STANDARD)
                return KeyEvent.KEY_LOCATION_STANDARD;
            if (loc == KeyLocation.LEFT)
                return KeyEvent.KEY_LOCATION_LEFT;
            if (loc == KeyLocation.RIGHT)
                return KeyEvent.KEY_LOCATION_RIGHT;
            if (loc == KeyLocation.KEYPAD)
                return KeyEvent.KEY_LOCATION_NUMPAD;
            return 0;
        }
         
        private int getModifiersEx() {
            int modifiersEx = 0x00;
            if (Keyboard.isPressed(Keyboard.Modifier.ALT))
                modifiersEx |= InputEvent.ALT_DOWN_MASK;
            if (Keyboard.isPressed(Keyboard.Modifier.CTRL))
                modifiersEx |= InputEvent.CTRL_DOWN_MASK;
            if (Keyboard.isPressed(Keyboard.Modifier.META))
                modifiersEx |= InputEvent.META_DOWN_MASK;
            if (Keyboard.isPressed(Keyboard.Modifier.SHIFT))
                modifiersEx |= InputEvent.SHIFT_DOWN_MASK;
            return modifiersEx;
        }
         
        private int getButtons() {
            int buttons = 0x00;
            if (Mouse.isPressed(Mouse.Button.LEFT))
                buttons |= MouseEvent.BUTTON1;
            if (Mouse.isPressed(Mouse.Button.MIDDLE))
                buttons |= MouseEvent.BUTTON2;
            if (Mouse.isPressed(Mouse.Button.RIGHT))
                buttons |= MouseEvent.BUTTON3;             
            return buttons;
        }
    }
}
