package org.interactivemesh.scala.j3d.samples.distortstring

// Java 
import java.awt.{Color, Dimension, Font, Graphics2D, GraphicsDevice, GraphicsEnvironment, Point}

// Scala Swing
import scala.swing.{Alignment, BorderPanel, Button, BoxPanel, GridPanel, 
  Label, MenuItem, Orientation, RootPanel, Separator, Slider, Swing}
import scala.swing.event.{ButtonClicked, MouseClicked, MouseEntered, 
  MouseExited, MousePressed, MouseReleased, UIElementResized, ValueChanged}    

// ImScalaSwing API 1.0, see http://www.interactivemesh.org/testspace/j3dmeetsscala.html
import org.interactivemesh.scala.swing.{LayeredPane, PopupMenu}

/*
 * DistortStringPanel.scala
 *
 * Version: 1.2
 * Date: 2011/05/26
 *
 * Copyright (c) 2010-2011
 * August Lammersdorf, InteractiveMesh e.K.
 * Kolomanstrasse 2a, 85737 Ismaning
 * Germany / Munich Area
 * www.InteractiveMesh.com/org
 *
 * Please create your own implementation.
 * This source code is provided "AS IS", without warranty of any kind.
 * You are allowed to copy and use all lines you like of this source code
 * without any copyright notice,
 * but you may not modify, compile, or distribute this 'DistortStringPanel.scala'.
 *
 */

private final class DistortStringPanel(
    gd: GraphicsDevice, 
    screenSize: Dimension,
    private[distortstring] var frame: App) 
      extends BoxPanel(Orientation.Vertical) {
  
  private val bgColor = new Color(0.0f, 0.4f, 0.8f)
  private val fgColor = new Color(1.0f, 1.0f, 1.0f)
  private val stColor = new Color(0.8f, 0.8f, 0.8f)
  
  private val screenHeight = screenSize.height

  // screenHeight >= 1200
  private var textFontSize = 18
  private var titleFontSize = 30
  private var borderBLR = 50
  private var borderT = 80
  private var space = 14

  // screenHeight  < 1024
  if (screenHeight < 1024) {
    textFontSize = 14
    titleFontSize = 24
    borderBLR = 30
    borderT = 50
    space = 5
  }
  // 1024 <= screenHeight < 1200
  else if (screenHeight < 1200) {
    textFontSize = 16
    titleFontSize = 26
    borderBLR = 40
    borderT = 60
    space = 8
  }

  private val symbolFont: Font = new Font("Lucida Sans", Font.PLAIN, titleFontSize)
  private val titleFont: Font = new Font("Dialog", Font.PLAIN, titleFontSize)
  private val textFont: Font = new Font("Dialog", Font.PLAIN, textFontSize)

  // Lightweight popup menu
  private object popupMenu extends PopupMenu {	
	  
    border = Swing.LineBorder(Color.WHITE)
	  
    private abstract class PopupMenuItem(s: String) extends MenuItem(s) {
      font = textFont
      background = bgColor
      foreground = fgColor
      
      def itemAction: Unit
      
      listenTo(mouse.clicks)
      reactions += {
      	case e: MouseReleased => itemAction
      	case _ =>
      }
    }
	  
    contents += new PopupMenuItem("Hide/Show controls") {
      def itemAction = universe.hideShowPanel(!controlPanelHided)
    }
    
    if (frame.isInstanceOf[DistortStringFrame.FrameApp]) {
      contents += new Separator
      contents += new PopupMenuItem("Switch window") {
    	def itemAction = frame.asInstanceOf[DistortStringFrame.FrameApp].switchAppFrame
      }
      contents.addSeparator
      contents += new PopupMenuItem("Exit application") {
        def itemAction = frame.asInstanceOf[DistortStringFrame.FrameApp].closeApp
      }
    }
  }

	
  // Universe
  private val universe = new DistortStringUniverse(this)

  // SJCanvas3D  
  private val sjCanvas3D = new DistortStringSJCanvas3D(
      gd, 
      new javax.media.j3d.GraphicsConfigTemplate3D, 
      this) {	  
	  
    // Initial values
    fraction = 0.75f
    topOpacity = 0.1f
    bottomOpacity = 0.8f
    topColorRGB = Color.BLACK
    bottomColorRGB = Color.BLACK
    
    listenTo(this.mouse.clicks, this.mouse.moves)
	reactions += {	 	  
	  case e: MouseEntered =>
	    if (controlPanelHided && e.point.y > this.size.height-10) {
	      universe.hideShowPanel(false)
	    }
      case e: MouseClicked =>
        if (e.peer.getButton == java.awt.event.MouseEvent.BUTTON3) {
          popupMenu.show(e.source, e.point.x+3, e.point.y)
        }
    }
  }


  // Control panel
  private object controlPanel extends BoxPanel(Orientation.Horizontal) {
	  
	opaque = false
	
    def bounds_=(bnd: (Int, Int, Int, Int)) {
	  this.peer.setBounds(bnd._1, bnd._2, bnd._3, bnd._4)
    }
	
    // FPS updated from universe
    private[DistortStringPanel] val fpsLabel = new TLabel("0") {
      tooltip = "Frames per second"
    }
  
    // Texture radio-buttons
    import universe.TexMode._
    private val textureGroup = new RadioGroup
    private[DistortStringPanel] val textureButtons = Array(
      new SRadioLabel("G1") { 
        radioGroup = textureGroup
        tooltip = "Spherical environment texture"
        def action = universe.varyGoldTexture(Gold1)
      },
      new SRadioLabel("G2") { 
        radioGroup = textureGroup
        tooltip = "Distorted object linear texture"
        def action = universe.varyGoldTexture(Gold2)
      },
      new SRadioLabel("G3") { 
        radioGroup = textureGroup
        tooltip = "Object linear texture"
        def action = universe.varyGoldTexture(Gold3)
      },
      new SRadioLabel("W1") { 
        radioGroup = textureGroup
        tooltip = "Distorted object linear procedural texture shader"
        def action = universe.varyWoodTexture(Wood1)
      },
      new SRadioLabel("W2") { 
        radioGroup = textureGroup
        tooltip = "Object linear procedural texture shader"
        def action = universe.varyWoodTexture(Wood2)
      }
    )
    textureGroup.select(textureButtons(0))
	
    // Slider size
	val opacLabel = new TLabel("Bottom Opacity")
	val slider = new Slider
    val sliderDim = slider.preferredSize
    sliderDim.width = (opacLabel.preferredSize.width * 1.1).asInstanceOf[Int]
        
	contents.append(
			
      // Viewpoints: Front, Side
	  new BoxPanel(Orientation.Vertical) {
	    opaque = false
	    
	    contents.append(
          new SActionLabel("Front") { 
    	    tooltip = "Front viewpoint"
	        def action = universe.vantagePoint(text)    
	      },
	       new SActionLabel("Side") { 
		    tooltip = "Side viewpoint"
		    def action = universe.vantagePoint(text)
		  }
	    )
	  },
	  
	  Swing.HStrut(space),
	  
	  // Animation start/stop
      new GridPanel(2, 2) {  
	    opaque = false
        hGap = 3
        vGap = 3
       
        // Distort speed
        val distortSlider = new Slider { 
	      min = 1
	      max = 100
	      value = 30
	      opaque = false
	      preferredSize = sliderDim
	      listenTo(this)
	      reactions += {
            case ValueChanged(_) =>
              universe.speedDistort(value)
	      }
        }	
        // Bounce speed
        val bounceSlider = new Slider {
	      min = 1
	      max = 100
	      value = 30
	      opaque = false
	      preferredSize = sliderDim
	      listenTo(this)
	      reactions += {
            case ValueChanged(_) =>
              universe.speedRebound(value)
	      }
        }	
    
        contents += ( 
          new SToggleLabel("Distortion") { 
	 	    tooltip = "Start / stop 3D string"
	 	    foreground = selectColor
	 	    selected = true
	        def onAction = universe.startStopDistort(true)
	        def offAction = universe.startStopDistort(false)
	      },	    		
	      new SToggleLabel("Rebound") { 
	 	    tooltip = "Start / stop 2D string"
	 	    foreground = selectColor
	 	    selected = true
	        def onAction = universe.startStopRebound(true)
	        def offAction = universe.startStopRebound(false)
	      },
	      distortSlider,
	      bounceSlider
        )
      },
      
	  Swing.HStrut(space),

	  // Slider reflection
	  
      new GridPanel(2, 3) {  
	    opaque = false
        hGap = 3
        vGap = 3
        tooltip = "Reflection"
       
        // Fraction
        val fractionSlider = new Slider {
	      min = 0
	      max = 100
	      value = 75
	      opaque = false
	      preferredSize = sliderDim
        }	
        // Top Opacity
        val topOpacitySlider = new Slider {
	      min = 0
	      max = 100
	      value = 10
	      opaque = false
	      preferredSize = sliderDim
        }	
        // Bottom Opacity
        val botOpacitySlider = new Slider {
	      min = 0
	      max = 100
	      value = 80
	      opaque = false
	      preferredSize = sliderDim
        }
    
        listenTo(fractionSlider, topOpacitySlider, botOpacitySlider)
        reactions += {
          case ValueChanged(`fractionSlider`) =>
            if (!fractionSlider.adjusting) {
            	sjCanvas3D.fraction = fractionSlider.value/100.0f
            	val size = sjCanvas3D.onscreenJPanel.size
            	universe.setupBgImage(size.width, size.height - sjCanvas3D.fractionHeight)
            }
          case ValueChanged(`topOpacitySlider`) =>
            sjCanvas3D.topOpacity = topOpacitySlider.value/100.0f
          case ValueChanged(`botOpacitySlider`) =>
            sjCanvas3D.bottomOpacity = botOpacitySlider.value/100.0f
        }
        
        contents += (
          new TLabel("Fraction"),
          new TLabel("Top Opacity"),
          opacLabel,
      
          fractionSlider,
          topOpacitySlider,
          botOpacitySlider
        )
      },
      
	  Swing.HStrut(space),
	  
	  // Textures: Gold, Wood 
      new BoxPanel(Orientation.Vertical) {
	    opaque = false   
        contents += ( 
          // Gold
	      new BoxPanel(Orientation.Horizontal) {
		    opaque = false   
	        contents += (
	          textureButtons(0),
	          Swing.HStrut(space),
	          textureButtons(1),
	          Swing.HStrut(space),
	          textureButtons(2)
	        )		          
	        override def paintComponent(g2d: Graphics2D) {               
	            super.paintComponent(g2d)                
	            val size = this.size
	            val step = (size.width-2*space)/3
	            g2d.setColor(Color.WHITE)              
	            // Vertical lines
	            g2d.drawLine(step+space/2, 4, step+space/2, size.height-4)
	            g2d.drawLine(2*step+3*space/2, 4, 2*step+3*space/2, size.height-4)
	        }
	      },
          // Wood
	      new BoxPanel(Orientation.Horizontal) {
		    opaque = false   
	        contents += (
	          textureButtons(3), 
	          Swing.HStrut(space),
	          textureButtons(4)
	        )	              
	        override def paintComponent(g2d: Graphics2D) {               
	          super.paintComponent(g2d)                
	          val size = this.size
	          g2d.setColor(Color.WHITE)              
	          // Vertical line
	          g2d.drawLine(size.width/2, 4, size.width/2, size.height-4)
	        }
	      }
        )
        override def paintComponent(g2d: Graphics2D) {               
          super.paintComponent(g2d)                
          val size = this.size
          g2d.setColor(Color.WHITE)              
          // Horizontal line
          g2d.drawLine(0, size.height/2, size.width, size.height/2)
	    }
      },
           
	  Swing.HStrut(space),
	  
	  // F P S
      new BoxPanel(Orientation.Vertical) {
	    opaque = false   
        contents += ( 
          new TLabel("FPS") { tooltip = "Frames per second" },
          fpsLabel
        )
      },
	  
 	  Swing.HStrut(space),
     
	  // Hide controls
      new BoxPanel(Orientation.Vertical) {
	    opaque = false   
        contents += ( 
          new SActionLabel("\u21e9") { 
        	font = symbolFont
            tooltip = "Hide controls"
            def action = universe.hideShowPanel(true)
          }
        )
      }
    )
  }
  
  // Hide/Show control panel
  
  private val controlsWidth = controlPanel.preferredSize.width
  private val controlsHeight = controlPanel.preferredSize.height 
  private val controlsBorderDist = (controlsHeight*1.5).asInstanceOf[Int]
  // Control visibility
  private var controlPanelHided = false
  // Control panel y-location
  private var controlsLocY: Int = 0
  // Control panel y-offset
  private var controlsOffsetY: Int = 0
  // Called from universe, y = [0, 1]
  private[distortstring] def hideShowControls(y: Float): Unit = {
	
	controlsOffsetY = (controlsBorderDist * y + 0.5f).asInstanceOf[Int]
	
	controlPanelHided = (controlsOffsetY >= controlsBorderDist)
	
	if (controlPanelHided) controlPanel.visible = false
	else if (!controlPanel.visible) controlPanel.visible = true
	
	val bounds = controlPanel.bounds
	controlPanel.bounds = (bounds.x, controlsLocY+controlsOffsetY, bounds.width, bounds.height)
    
	// Will be repainted in regular repainting loop
  }
  
  import LayeredPane._
  
  // 3D scene and controls
  private val layeredPane = new LayeredPane { self =>
		  
    layout(sjCanvas3D) = new LayerConstraints(Layer.Default)
    layout(controlPanel) = new LayerConstraints(Layer.Palette)
    
    listenTo(self)
    reactions += {
      case UIElementResized(`self`) =>
       					
        val size = self.size
        sjCanvas3D.bounds = (0, 0, size.width, size.height)
        // Resize background image
        universe.setupBgImage(size.width, size.height - sjCanvas3D.fractionHeight)
        
        controlsLocY = size.height-controlsBorderDist
        controlPanel.bounds = ( 
          (size.width-controlsWidth)/2, 
          controlsLocY + controlsOffsetY, 
          controlsWidth, 
          controlsHeight
        )
      
        self.revalidate
        self.repaint
	}   
  }
  
  // 
  // 1. Application panel 
  //
  contents.append(   		
    // Title
    new BoxPanel(Orientation.Horizontal) {
      background = bgColor
		
	  contents.append(
	    Swing.HStrut(borderBLR),
	    new SLabel("Java 3D meets Scala : Composite 2D & 3D Rendering") {
		  font = titleFont
		  foreground = stColor
		  xLayoutAlignment = 0.0
		  horizontalAlignment = Alignment.Left
		  preferredSize = new Dimension(this.preferredSize.width, borderT)
		},
        Swing.HGlue
      )
    },    		
    // 3D scene, controls
    new BorderPanel {   
      background = bgColor
	  border = Swing.CompoundBorder(
	    Swing.EmptyBorder(0, borderBLR, borderBLR, borderBLR),
	    Swing.LineBorder(stColor, 3)
	  )
	  
	  layout(layeredPane) = BorderPanel.Position.Center
    }
  )
  
  // 2. Add canvas to universe
  universe.addCanvasPanel(sjCanvas3D)
	
  // GLSL shader available ?
  if (!universe.isGLSLavailable) {
	controlPanel.textureButtons(3).tooltip ="GLSL not available !"
	controlPanel.textureButtons(4).tooltip ="GLSL not available !"
	controlPanel.textureButtons(3).enabled = false
	controlPanel.textureButtons(4).enabled = false
  }
  
  // Print properties
  frame.printSystemProps
  universe.printJava3DProps

  // GUI update: frame per second
  private[distortstring] def updateFPS(fps: Int) = controlPanel.fpsLabel.text = fps.toString
  // Cleanup
  private[distortstring] def closePanel = universe.closeUniverse
	
  
  //
  // Nested GUI classes : Lightweight 'buttons' for a transparent overlay
  //
  
  private[DistortStringPanel] class SLabel(s: String) extends Label(s) {
    
    protected val enterColor = Color.ORANGE
    protected val pressColor = Color.RED       
    protected val selectColor = new Color(0.0f, 0.8f, 0.8f)  //(0.0f, 1.0f, 0.6f)            
	private   var isPressed = false
	private   var isSelected = false
        
    font = textFont
    foreground = fgColor
    xLayoutAlignment = 0.5
	 
    protected def pressed: Boolean = isPressed
    protected def pressed_=(p: Boolean) = isPressed = p

    protected def selected: Boolean = isSelected
    protected def selected_=(s: Boolean) = isSelected = s

	protected def contains(p: Point): Boolean = {
	  peer.contains(p.x, p.y)
	}
  }
  
  private[DistortStringPanel] class TLabel(s: String) extends SLabel(s) {
	def this() = this("")
	
	foreground = stColor
  }

  private[DistortStringPanel] abstract class SActionLabel(s: String) extends SLabel(s) {
	  
    listenTo(mouse.clicks, mouse.moves)
	reactions += {	 	  
	  case e: MouseEntered =>
	    if (!pressed) foreground = enterColor
	  case e: MouseExited =>
	    if (!pressed) foreground = fgColor
	  case e: MousePressed =>
	    pressed = true
        foreground = pressColor
	  case e: MouseReleased =>
        pressed = false        
        if (contains(e.point)) {
        	foreground = enterColor
        	action
        }
        else {
        	foreground = fgColor
        }
	}
    
    def action
  }
  
  private[DistortStringPanel] abstract class SToggleLabel(s: String) extends SLabel(s) {
	  
    listenTo(mouse.clicks, mouse.moves)
	reactions += {	 	  
	  case e: MouseEntered =>
	    if (!pressed) foreground = enterColor
	  case e: MouseExited =>
	    if (!pressed) {
	    	if (selected) foreground = selectColor
	    	else foreground = fgColor
	    }
	  case e: MousePressed =>
	    pressed = true
        foreground = pressColor
	  case e: MouseReleased =>
        pressed = false                
        if (contains(e.point)) {
        	selected = !selected
        	if (selected) onAction
        	else offAction
        }
        
    	if (selected) foreground = selectColor
    	else foreground = fgColor
	}
    
    def onAction
    def offAction
  }
  
  private[DistortStringPanel] abstract class SRadioLabel(s: String) extends SLabel(s) {
	  
	private var rGroup: RadioGroup = null
	
	def radioGroup: RadioGroup = rGroup
	def radioGroup_=(rg: RadioGroup) = rGroup = rg
	
	// Note: Disabling a lightweight component does not prevent it from receiving MouseEvents. 
	override def enabled_=(b: Boolean) {
	  super.enabled = b 
	  if (b) listenTo(mouse.clicks, mouse.moves)
	  else deafTo(mouse.clicks, mouse.moves)
	}
	  
    private[DistortStringPanel] def select(b: Boolean) {
	  selected = b
      if (selected) foreground = selectColor
      else foreground = fgColor
    }
    
    listenTo(mouse.clicks, mouse.moves)
	reactions += {	 	  
	  case e: MouseEntered =>
	    if (!pressed && !selected) foreground = enterColor
	  case e: MouseExited =>
	    if (!pressed && !selected) {
	    	foreground = fgColor
	    }
	  case e: MousePressed =>
	    if (!selected) {
	      pressed = true
          foreground = pressColor
	    }
	  case e: MouseReleased =>
	    if (!selected) {
          pressed = false                
          if (contains(e.point)) {
        	rGroup.select(SRadioLabel.this)
        	action
          }
    	  else foreground = fgColor
	    }   
	}
    
    def action
  }
  
  private[DistortStringPanel] final class RadioGroup() {
	  	  
	private var selected: SRadioLabel = null
	
	def select(rl: SRadioLabel) {
	  if (selected != null)
		  selected.select(false)
	  selected = rl
	  selected.select(true)
    }
  }
}
