package org.interactivemesh.scala.j3d.samples.propeller

// Java
import java.awt.{Color, Dimension, Font, Graphics2D, GraphicsDevice, Point}

// Scala Swing
import scala.swing.{Alignment, BorderPanel, ButtonGroup, BoxPanel, 
  Component, GridPanel, Label, MenuItem, Orientation, Publisher, 
  RadioButton, Separator, Slider, Swing}
import scala.swing.event.{ButtonClicked, MouseClicked, MouseReleased, UIElementResized, ValueChanged}

// ImScalaSwing API 1.0, see http://www.interactivemesh.org/testspace/j3dmeetsscala.html
import org.interactivemesh.scala.swing.{InternalDefaultDesktopManager, 
  InternalDesktopPane, InternalFrame, LayeredPane, PopupMenu}
import org.interactivemesh.scala.swing.InternalDesktopPane.DragMode

// ScalaCanvas3D API 1.0, see http://www.interactivemesh.org/testspace/j3dmeetsscala.html
import org.interactivemesh.scala.swing.j3d.SCanvas3D

/*
 * PropellerPanel.scala
 *
 * Version: 1.2
 * Date: 2011/05/26
 *
 * Copyright (c) 2010-2011
 * August Lammersdorf, InteractiveMesh e.K.
 * Kolomanstrasse 2a, 85737 Ismaning
 * Germany / Munich Area
 * www.InteractiveMesh.com/org
 *
 * Please create your own implementation.
 * This source code is provided "AS IS", without warranty of any kind.
 * You are allowed to copy and use all lines you like of this source code
 * without any copyright notice,
 * but you may not modify, compile, or distribute this 'PropellerPanel.scala'.
 *
 */

private final class PropellerPanel(
    gd: GraphicsDevice, 
    screenSize: Dimension,
    private[propeller] var frame: App) 
      extends BoxPanel(Orientation.Vertical) {
	
  private val screenHeight = screenSize.height

  // screenHeight >= 1200
  private var textFontSize = 20
  private var titleFontSize = 48
  private var borderBLR = 50
  private var borderT = 80
  private var space = 14

  // screenHeight  < 1024
  if (screenHeight < 1024) {
    textFontSize = 16
    titleFontSize = 28
    borderBLR = 36
    borderT = 50
    space = 10
  }
  // 1024 <= screenHeight < 1200
  else if (screenHeight < 1200) {
    textFontSize = 18
    titleFontSize = 42
    borderBLR = 40
    borderT = 60
    space = 12
  }

  private val titleFont: Font = new Font("Serif", Font.ITALIC, titleFontSize)
  private val textFont: Font = new Font("Dialog", Font.BOLD, textFontSize)
  
  private val fgColor = new Color(0.0f, 0.0f, 0.0f)

  // Background image, also used in 3D scene   
  private val imageUrl: java.net.URL = this.getClass().getResource("resources/bg.jpg")
  private val bgImage: java.awt.image.BufferedImage = 
    try {
      javax.imageio.ImageIO.read(imageUrl)
    }
    catch {
      case e: java.io.IOException => null
    }
  
  // Universe
  private val universe = new PropellerUniverse(PropellerPanel.this, gd)
  
  import universe.ViewPoint._
  
  // Heavyweight popup menu  
  PopupMenu.defaultLightWeightPopupEnabled = false
  
  private object popupMenu extends PopupMenu {	
	  
    border = Swing.LineBorder(Color.WHITE)
	  
    // Background image
    background = new Color(0,0,0,0)
    val imageW = bgImage.getWidth
    val imageH = bgImage.getHeight
    override def paintComponent(g2d: Graphics2D) {
      peer match {
        case peer: SuperMixin => super.paintComponent(g2d)
          if (bgImage ne null) {
            val size = peer.size
            g2d.drawImage(
              bgImage, 
              0, 0, size.width, size.height, 
              imageW-size.width, imageH-size.height, imageW, imageH, null
            )                          
          }
        case _ => {}
      }
    }
	
    private var vp = Front
	  
    private[PropellerPanel] def show(invoker: Component, p: Point, viewpoint: ViewPoint) {
      vp = viewpoint
      show(invoker, p.x+3, p.y)
    }

    private[PropellerPanel] def frameMaximum(b: Boolean) {
      layoutItem.enabled = !b
    }
    
    private abstract class PopupMenuItem(s: String) extends MenuItem(s) {
      font = textFont
      foreground = fgColor
      opaque = false
      
      def itemAction: Unit
      
      listenTo(mouse.clicks)
      reactions += {
      	case e: MouseReleased => if (this.enabled) itemAction
      	case _ =>
      }
    }
	
    private val layoutItem = new PopupMenuItem("Lay out frames") {
    	def itemAction = desktopPane.setupLayeredPane
    }
	
    contents += new PopupMenuItem("Set viewpoint") {
      def itemAction = universe.vantagePoint(vp)
    }
    contents += new Separator
    contents += layoutItem
    contents += new Separator
    contents += new PopupMenuItem("Stop rotation") {
      def itemAction = controlPanel.rotationSlider.value = 0
    }
    if (frame.isInstanceOf[PropellerFrame.FrameApp]) {
      contents += new Separator
      contents += new PopupMenuItem("Switch window") {
    	def itemAction = frame.asInstanceOf[PropellerFrame.FrameApp].switchAppFrame    	  
      }
      contents.addSeparator
      contents += new PopupMenuItem("Exit application") {
        def itemAction = frame.asInstanceOf[PropellerFrame.FrameApp].closeApp
      }
    }
  }
  
  // Internal SCanvas3D wrapper type
  private trait InternalSCanvas3D extends Publisher {
	  
    // Canvas/Viewpoint
    private[PropellerPanel] val canvas3d: SCanvas3D
    protected val vp: ViewPoint

    // Position & size in layered pane
    def bounds_=(bnd: (Int, Int, Int, Int))

    // Show popup menu
    reactions += {
      case e: MouseClicked =>
        if (e.peer.getButton == java.awt.event.MouseEvent.BUTTON3) {
          popupMenu.show(e.source, e.point, vp)
        }
    }
  }
    
  // Internal frame with SCanvas3D
  private final class SCanvas3DFrame(val vp: ViewPoint) extends InternalFrame with InternalSCanvas3D { 
	  
    override private[PropellerPanel] val canvas3d = new SCanvas3D(universe.canvas3D(vp)) 
	
    listenTo(canvas3d.canvasMouseClick)

    title = vp.toString

    border = Swing.LineBorder(Color.WHITE, 2)

    iconifiable = true
    maximizable = true
    resizable = true
    visible = true

    contents = canvas3d
  }
  
  // Map: ViewPoint -> InternalSCanvas3D
  private val intSCanvas3Ds = Map(
    Piston -> new SCanvas3DFrame(Piston),
    Rod    -> new SCanvas3DFrame(Rod),
    OnRod  -> new SCanvas3DFrame(OnRod),
              // Internal desktop background with SCanvas3D : not of type InternalFrame !
    Front  -> new BorderPanel with InternalSCanvas3D { 
	  
      override private[PropellerPanel] val canvas3d = new SCanvas3D(universe.canvas3D(Front))
      override protected val vp = Front

      listenTo(canvas3d.canvasMouseClick)
	
      def bounds_=(bnd: (Int, Int, Int, Int)) {
        this.peer.setBounds(bnd._1, bnd._2, bnd._3, bnd._4)
      }
    
      layout(canvas3d) = BorderPanel.Position.Center
    }
  )
  
  // Control palette
  private val intFramePalette = new InternalFrame {  
    contents = controlPanel
    peer.putClientProperty("JInternalFrame.isPalette", true)
    visible = true
  }
    
  //
  // Control panel
  //
  private object controlPanel extends BoxPanel(Orientation.Horizontal) { 
	      	
    // Heavyweight tooltip component (seems to work here also without setting to false ?)
    javax.swing.ToolTipManager.sharedInstance.setLightWeightPopupEnabled(false)

    border = Swing.EmptyBorder(5, 5, 5, 5)
	
    // Propeller rotation
    private val rotNameLabel = new Label("Rotation") {
      font = textFont
      horizontalAlignment = Alignment.Left
    }
    private[PropellerPanel] val rotRPMLabel = new Label("000 RPM") {
      font = textFont
      horizontalAlignment = Alignment.Right
    }
    private[PropellerPanel] val rotationSlider = new Slider {
      opaque = false  
      min = 0
      max = 100
      value = 0
      xLayoutAlignment = 0.5
      preferredSize = new Dimension((rotNameLabel.preferredSize.width*3).asInstanceOf[Int], preferredSize.height)
      minimumSize = preferredSize
      
      listenTo(this)
      reactions += {
        case ValueChanged(_) =>
          universe.rotationSpeed(value)
      }
    }    
	
    // FPS: updated by universe
    private[PropellerPanel] val fpsLabel = new Label() {
      font = textFont
      text = "0"
      horizontalAlignment = Alignment.Center
    }
        
    contents.append(
			
      // Rotation
      new BoxPanel(Orientation.Vertical) {
        opaque = false
        tooltip = "Rotations per minute"
        contents += ( 
          new GridPanel(1, 2) {
            hGap = space
        	opaque = false
        	contents += (rotNameLabel, rotRPMLabel)
            maximumSize = this.preferredSize
          },          
          rotationSlider
        )
      },	
			
      Swing.HStrut(space),

      // F P S
      new GridPanel(2, 1) {
    	opaque = false
    	tooltip = "Frames per second while propeller is rotating"
    	contents += (
          new Label {
            font = textFont
            text = " FPS "
          }, 
          fpsLabel
        )
        maximumSize = this.preferredSize
      },
      
      Swing.HStrut(space),
      
      // Drag mode
      new BoxPanel(Orientation.Vertical) {
    	opaque = false  
    	
        val live = new RadioButton("Live") {
    	  font = textFont
    	  opaque = false
    	  selected = true
    	  tooltip = "Drag mode 'Live'"
    	}
        val outline = new RadioButton("Outline") {
          font = textFont
          opaque = false
          tooltip = "Drag mode 'Outline'"
        }
        val mutex = new ButtonGroup(live, outline)
    
        contents ++= mutex.buttons
        
        listenTo(live, outline)
        reactions += {
          case ButtonClicked(`live`) => 
            desktopPane.dragMode = DragMode.Live
          case ButtonClicked(`outline`) => 
            desktopPane.dragMode = DragMode.Outline
        }
      }
    )
	
    // Background image
    override def paintComponent(g2d: Graphics2D) {
      peer match {
        case peer: SuperMixin => super.paintComponent(g2d)
          if (bgImage ne null) {
            val size = controlPanel.size
            g2d.drawImage(
              bgImage, 
              0, 0, size.width, size.height, 
              bgImage.getWidth-size.width, bgImage.getHeight-size.height, bgImage.getWidth, bgImage.getHeight, null
            )
          }
        case _ =>
      }
    }
  }
 
  //
  // InternalDesktopPane & InternalDefaultDesktopManager
  //
  private object desktopPane extends InternalDesktopPane { 
	  	
    // InternalDefaultDesktopManager
    desktopManager = manager

    dragMode = DragMode.Live

    import LayeredPane._

    layout(intSCanvas3Ds.get(Front).get)  = new LayerConstraints(layer = 1)               // background

    layout(intSCanvas3Ds.get(Piston).get) = new LayerConstraints(layer = 2, position = 0) // canvas3d
    layout(intSCanvas3Ds.get(Rod).get)    = new LayerConstraints(position = 1, layer = 2) //   "
    layout(intSCanvas3Ds.get(OnRod).get)  = new LayerConstraints(layer = 2, position = 2) //   "
	
    layout(intFramePalette) = new LayerConstraints(layer = 3)                             // controls

    listenTo(this)
    reactions += {
      case UIElementResized(desktopPane) =>      					
        val size = desktopPane.size
        setupLayeredPane
    }
	
	// Layout
    def setupLayeredPane: Unit = {
    	
      val size = this.size
       
      val iframesHeight = size.height - borderBLR*2
    
      val iFrameWidth = (size.width * 0.3f).asInstanceOf[Int]
      val iframeHeight = ((iframesHeight - 2*space)/3.0f).asInstanceOf[Int]
    
      val frameX = borderBLR
      val frameY = borderBLR                
      
// TODO      
val isBounds = false
      
      if (isBounds) {
        // Background frame
        intSCanvas3Ds.get(Front).get.bounds  = (0, 0, size.width, size.height)
        // Internal canvas frames 
        intSCanvas3Ds.get(Piston).get.bounds = (frameX, frameY, iFrameWidth, iframeHeight)
        intSCanvas3Ds.get(Rod).get.bounds    = (frameX, frameY+iframeHeight+space, iFrameWidth, iframeHeight)
        intSCanvas3Ds.get(OnRod).get.bounds  = (frameX, frameY+(iframeHeight+space)*2, iFrameWidth, iframeHeight)
      }
      else {
        manager.setBoundsForFrame(intSCanvas3Ds.get(Front).get, 0, 0, size.width, size.height)
      
        manager.setBoundsForFrame(intSCanvas3Ds.get(Piston).get, frameX, frameY, iFrameWidth, iframeHeight)
        manager.setBoundsForFrame(intSCanvas3Ds.get(Rod).get, frameX, frameY+iframeHeight+space, iFrameWidth, iframeHeight)
        manager.setBoundsForFrame(intSCanvas3Ds.get(OnRod).get, frameX, frameY+(iframeHeight+space)*2, iFrameWidth, iframeHeight)
      }
      
      // Control panel
      val paletteWidth = intFramePalette.preferredSize.width
      val paletteHeight = intFramePalette.preferredSize.height  
      
      if (isBounds) {
        intFramePalette.bounds = (
          size.width - paletteWidth - borderBLR, 
          size.height - paletteHeight - borderBLR,
          paletteWidth,
          paletteHeight
        )
      }
      else {
        manager.setBoundsForFrame(intFramePalette, size.width - paletteWidth - borderBLR, size.height - paletteHeight - borderBLR, paletteWidth, paletteHeight)
      }
      
      this.revalidate()
      this.repaint()
    }
  }
  
  private object manager extends InternalDefaultDesktopManager {
	  
    // Keep Component within the area of desktop
    override def dragFrame(c: Component, newX: Int, newY: Int) {
		
      var x = newX
      var y = newY
		
      if (x < 0) x = 0
      else {
        if (x + c.size.width > desktopPane.size.width) {
          x = desktopPane.size.width - c.size.width
        }
      }
    
      if (y < 0) y = 0
      else {
        if (y + c.size.height > desktopPane.size.height) {
          y = desktopPane.size.height - c.size.height
        }
      }
   		
      super.dragFrame(c, x, y)
    }
	
    override def deiconifyFrame(f: InternalFrame) {
      if (f.maximum) {
        // Stop Java 3D rendering for non-visible canvas3ds
        intSCanvas3Ds foreach { i =>
          if (i._2 ne f) i._2.canvas3d.stopRenderer 
        }      
        
        popupMenu.frameMaximum(true)
      }

      super.deiconifyFrame(f)
    }

    override def iconifyFrame(f: InternalFrame) {
      if (f.maximum) {
        // Start Java 3D rendering for all canvas3ds
        intSCanvas3Ds foreach { i =>
          if (i._2 ne f) i._2.canvas3d.startRenderer 
        }      
        
        popupMenu.frameMaximum(false)
      }
	   
      super.iconifyFrame(f)
    }
	
    override def maximizeFrame(f: InternalFrame) {
                              
      popupMenu.frameMaximum(true)
      
      super.maximizeFrame(f)
      
      // Stop Java 3D rendering for non-visible canvas3ds
      intSCanvas3Ds foreach { i =>
        if (i._2 ne f) i._2.canvas3d.stopRenderer 
      }      
    }
    
    override def minimizeFrame(f: InternalFrame) {
    	
      // Start Java 3D rendering for all canvas3ds
      intSCanvas3Ds foreach { i =>
        if (i._2 ne f) i._2.canvas3d.startRenderer 
      }      
            
      super.minimizeFrame(f)
      
      popupMenu.frameMaximum(false)
    }
  }


  // GUI update: frame per second 
  private[propeller] def updateFPS(fps: Int) = controlPanel.fpsLabel.text = fps.toString
  private[propeller] def updateRPM(rpm: Int) = controlPanel.rotRPMLabel.text = rpm.toString+" RPM"
  // Cleanup
  private[propeller] def closePanel = universe.closeUniverse
	
  
  // Application panel
  
  border = Swing.EmptyBorder(0, borderBLR, borderBLR, borderBLR) 

  contents.append(   		
    // Title
    new BoxPanel(Orientation.Horizontal) {
      opaque = false
      contents.append(
        Swing.HStrut(borderBLR),	      
        new Label {
          font = titleFont
          foreground = Color.WHITE
          text = "Java 3D meets Scala :  H'weight meets L'weight"
          xLayoutAlignment = 0.5
          horizontalAlignment = Alignment.Center
          preferredSize = new Dimension(this.preferredSize.width, borderT)
        },
        Swing.HStrut(borderBLR)
      )
    },    		
    // Internal desktop
    new BorderPanel {
      border = Swing.BeveledBorder(Swing.Lowered)	  
      layout(desktopPane) = BorderPanel.Position.Center	  
    }
  )
  
  // Background image
  override def paintComponent(g2d: Graphics2D) {
    peer match {
      case peer: SuperMixin => super.paintComponent(g2d)
        if (bgImage ne null) {
    	  val size = peer.size
          g2d.drawImage(bgImage, 0, 0, size.width, size.height, null)
        }    	
      case _ => {}
    }
  }
  
  // Print properties
  frame.printSystemProps
  universe.printJava3DProps
  // Start value '0'
  updateRPM(0)
}

