
/*
 * LICENSE
 * 
 * JCanvas3DOBAbstract.java 
 * 
 * is a derived work of the Java 3D utility class "com.sun.j3d.exp.swing.JCanvas3D.java".
 * 
 * Redistribution and use are permitted according to the following license notice.
 * 
 * Version: 1.0
 * Date: 2009/03/09
 * 
 * Author:
 * August Lammersdorf, InteractiveMesh e.K.
 * Kolomanstrasse 2a, 85737 Ismaning
 * Germany / Munich Area
 * www.InteractiveMesh.com/org
 * 
*/

/*
 * $RCSfile: com.sun.j3d.exp.swing.JCanvas3D.java,v $
 *
 * Copyright (c) 2007 Sun Microsystems, Inc. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * - Redistribution of source code must retain the above copyright
 *   notice, this list of conditions and the following disclaimer.
 *
 * - Redistribution in binary form must reproduce the above copyright
 *   notice, this list of conditions and the following disclaimer in
 *   the documentation and/or other materials provided with the
 *   distribution.
 *
 * Neither the name of Sun Microsystems, Inc. or the names of
 * contributors may be used to endorse or promote products derived
 * from this software without specific prior written permission.
 *
 * This software is provided "AS IS," without a warranty of any
 * kind. ALL EXPRESS OR IMPLIED CONDITIONS, REPRESENTATIONS AND
 * WARRANTIES, INCLUDING ANY IMPLIED WARRANTY OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE OR NON-INFRINGEMENT, ARE HEREBY
 * EXCLUDED. SUN MICROSYSTEMS, INC. ("SUN") AND ITS LICENSORS SHALL
 * NOT BE LIABLE FOR ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF
 * USING, MODIFYING OR DISTRIBUTING THIS SOFTWARE OR ITS
 * DERIVATIVES. IN NO EVENT WILL SUN OR ITS LICENSORS BE LIABLE FOR
 * ANY LOST REVENUE, PROFIT OR DATA, OR FOR DIRECT, INDIRECT, SPECIAL,
 * CONSEQUENTIAL, INCIDENTAL OR PUNITIVE DAMAGES, HOWEVER CAUSED AND
 * REGARDLESS OF THE THEORY OF LIABILITY, ARISING OUT OF THE USE OF OR
 * INABILITY TO USE THIS SOFTWARE, EVEN IF SUN HAS BEEN ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGES.
 *
 * You acknowledge that this software is not designed, licensed or
 * intended for use in the design, construction, operation or
 * maintenance of any nuclear facility.
 *
 * $Revision: 1.10 $
 * $Date: 2007/04/11 02:08:56 $
 * $State: Exp $
 * 
 *  @author: Frederic 'pepe' Barachant
 *
 */

package com.interactivemesh.j3d.testspace.jcanvas3d;

import java.awt.Graphics;
import java.awt.event.MouseEvent;
import java.awt.image.DataBufferInt;

import javax.media.j3d.GraphicsConfigTemplate3D;

import javax.swing.event.MouseInputAdapter;

import com.interactivemesh.j3d.community.gui.JCanvas3DOBAbstract;

/**
 * TuxCubeGrayscaleJC3DOB.java
 *
 * Version: 1.0
 * Date: 2009/03/09
 *
 * Copyright (c) 2009
 * August Lammersdorf, InteractiveMesh e.K.
 * Kolomanstrasse 2a, 85737 Ismaning
 * Germany / Munich Area
 * www.InteractiveMesh.com/org
 *
 * Please create your own implementation.
 * This source code is provided "AS IS", without warranty of any kind.
 * You are allowed to copy and use all lines you like of this source code
 * without any copyright notice,
 * under consideration of the license notice above for the suberclass 'JCanvas3DOBAbstract',
 * but you may not modify, compile, or distribute this 'TuxCubeGrayscaleJC3DOB.java'.
 *
 */
final class TuxCubeGrayscaleJC3DOB extends JCanvas3DOBAbstract {

    //
    // Spot
    //
    
    private int[]   imageArray  =   null;  
    private int     imageLength =   0;
    
    private int     spotPercent =   22;
    
    private int     spotX       =   0;
    private int     spotY       =   0;
    private int     spotWidth   =   0;
    private int     spotHeight  =   0;

    private int     southLength =   0;
    private int     westWidth   =   0;
    private int     eastWidth   =   0;
    private int     northStart  =   0;

    private int[]   rGray       =   new int[256];
    private int[]   gGray       =   new int[256];
    private int[]   bGray       =   new int[256];
    
    private MouseInputAdapter mouseAdapter = null;
    
    //
    // Frames per second
    //
    
    private int     updatesPerSecond    =   2;
    private int     elapsedFrames       =   0;
    private int     frameCounter        =   0;
    private long    startTimePaint      =   System.nanoTime()/1000000;
    private long    endTimePaint        =   0;
    private long    frameDuration       =   0;
    
    private TuxCubeGrayscale universe   =   null;

    /**
     * Constructs and initializes a new TuxCubeGrayscaleJC3DOB object that Java 3D
     * can render into, using the specified template.
     * The screen device is obtained from
     * <code>GraphicsEnvironment.getLocalGraphicsEnvironment().getDefaultScreenDevice()</code>,
     * which might not be the one you should use if you are
     * in a multiscreen environment.
     *
     * @param template The template that will be used to construct a
     *        GraphicsConfiguration. The stereo and doublebuffer properties
     *        are forced to UNNECESSARY.
     */
    TuxCubeGrayscaleJC3DOB(GraphicsConfigTemplate3D template, TuxCubeGrayscale tuxuniverse) {
        super(template);
                
        this.universe = tuxuniverse;
        
        // Lookup tables : a little bit faster than       
        // gray = (int)(((argb >> 16) & 0xFF)*0.212671f + ((argb >> 8) & 0xFF)*0.715516f + (argb & 0xFF)*0.071169f) & 0xFF;

        for (int i=0; i < 256; i++) {
            rGray[i] = (int)(i * 0.212671);
            gGray[i] = (int)(i * 0.715516);
            bGray[i] = (int)(i * 0.071169);
        }
        
        // Spot movement
        mouseAdapter = new MouseInputAdapter() {
            
            int startX = 0;
            int startY = 0;
            int startSpotX = 0;
            int startSpotY = 0;
            
            public void mousePressed(MouseEvent e) {                
                startX = e.getX();
                startY = e.getY();
                startSpotX = spotX;
                startSpotY = spotY;
            }
            
            public void mouseDragged(MouseEvent e) {
                
                boolean reset = false;
                
                spotX = startSpotX + (e.getX()-startX);
                if (spotX < 0) {
                    spotX = 0;
                    reset = true;
                }
                else if ( (spotX + spotWidth) > imageWidth) {
                    spotX = imageWidth - spotWidth;
                    reset = true;
                }
                
                // yUp !!
                spotY = startSpotY - (e.getY()-startY);
                if (spotY < spotHeight) {
                    spotY = spotHeight;
                    reset = true;                    
                }
                else if ( spotY > imageHeight) {
                    spotY = imageHeight;
                    reset = true;
                }
                
                if (reset) {
                    startX = e.getX();
                    startY = e.getY();
                    startSpotX = spotX;
                    startSpotY = spotY;
                }
                
                moveSpot();
                universe.repaintView();
            }
        };
    }
    
    // Enable spot movement
    void setSpotEnabled(boolean enable) {
        if (enable) {
            this.addMouseListener(mouseAdapter);
            this.addMouseMotionListener(mouseAdapter);
        }
        else {
            this.removeMouseListener(mouseAdapter);
            this.removeMouseMotionListener(mouseAdapter);
        }
    }
    
    // View.repaint() needed
    void setSpotSize(int percent) {
        if (percent > 100)
            percent = 100;
        else if (percent < 0)
            percent = 0;
        
        spotPercent = percent;
        
        resizeSpot();
        universe.repaintView();
    }
    
    // Called when panel or spot is resized: spot will be centered!
    private void resizeSpot() {
        
        int length = (int) (Math.max(imageWidth, imageHeight) * spotPercent / 100.0f + 0.5f);
        spotWidth = Math.min(length, imageWidth);
        spotHeight = Math.min(length, imageHeight);
        
        // yUp !!
        
        int southLines = (imageHeight - spotHeight) / 2;
        int northLines = imageHeight - southLines - spotHeight;
        
        southLength = imageWidth * southLines;
        
        westWidth = (imageWidth - spotWidth) / 2;
        eastWidth = imageWidth - westWidth - spotWidth;

        northStart = imageLength - imageWidth * northLines;
        
        spotX = westWidth;
        spotY = southLines + spotHeight;
    }
    
    //
    private void moveSpot() {
        
        // northLength, westWidth, eastWidth, southStart
        
        northStart  = imageWidth * spotY;
        southLength = imageWidth * (spotY - spotHeight);
        westWidth   = spotX;
        eastWidth   = imageWidth - westWidth - spotWidth;
    }
    
    /**
     * Creates new off-screen buffers of the given size. This method is called 
     * internally whenever this panel is added to a parent or is resized. 
     * <p>
     * Subclasses should call and/or override this method according to its
     * individual needs. In case of overriding calling
     * <code>super.createOffScreenBuffer(canvasWidth, canvasHeight)</code> 
     * has to be the last thing to do.</p>
     * @param width the width of the off-screen buffers to create
     * @param height the height of the off-screen buffers to create
     */
    @Override
    protected void createOffScreenBuffer(int canvasWidth, int canvasHeight) {
        
        // Nothing to do 

        // At last !!
        super.createOffScreenBuffer(canvasWidth, canvasHeight);
    }
    
    /**
     * Callback used to allow an overriding subclass to execute individual code
     * when new off-screen buffers were created. 
     * <p>
     * This method is called internally by the event-dispatching thread (EDT)
     * and should not be called by applications. 
     * </p>
     */
    @Override
    protected void offScreenBufferCreated() {
        
        // Get the data buffer from the new paintImage
        imageArray = ((DataBufferInt)paintImage.getRaster().getDataBuffer()).getData();   
        imageLength = imageArray.length;
        
        // Spot
        resizeSpot();
        
    }
    
    /**
     * Callback used to allow an overriding subclass to execute individual code
     * when the off-screen buffers were copied.
     * <p>
     * This method is called internally by the event-dispatching thread (EDT)
     * and should not be called by applications. 
     * </p>
     */
    @Override
    protected void offScreenBufferCopied() {
        
        int argb = 0;
        int gray = 0;
        
        int i = 0;
        int step = 0;
        
        // yUp !!

        // Border south
        for (i=0; i < southLength; i++) {
            argb = imageArray[i];
//            gray = (int)(((argb >> 16) & 0xFF)*0.212671f + ((argb >> 8) & 0xFF)*0.715516f + (argb & 0xFF)*0.071169f) & 0xFF;
            gray = (int)( rGray[((argb >> 16) & 0xFF)] + gGray[((argb >> 8) & 0xFF)] + bGray[(argb & 0xFF)]) & 0xFF;
            imageArray[i] = (argb & 0xFF000000) | (gray << 16) | (gray <<  8) | gray;
        }
        
        // Border west
        i = southLength;
        step = spotWidth + eastWidth;
        for (int j=0; j < spotHeight; j++) {
            for (int k=0; k < westWidth; k++) {
                argb = imageArray[i];
//                gray = (int)(((argb >> 16) & 0xFF)*0.212671f + ((argb >> 8) & 0xFF)*0.715516f + (argb & 0xFF)*0.071169f) & 0xFF;
                gray = (int)( rGray[((argb >> 16) & 0xFF)] + gGray[((argb >> 8) & 0xFF)] + bGray[(argb & 0xFF)]) & 0xFF;
                imageArray[i] = (argb & 0xFF000000) | (gray << 16) | (gray <<  8) | gray;
                i++;
            }
            
            i += step; 
        }        
        
        // Border east
        i = southLength + westWidth + spotWidth;
        step = westWidth + spotWidth;
        for (int j=0; j < spotHeight; j++) {
            for (int k=0; k < eastWidth; k++) {                
                argb = imageArray[i];
//                gray = (int)(((argb >> 16) & 0xFF)*0.212671f + ((argb >> 8) & 0xFF)*0.715516f + (argb & 0xFF)*0.071169f) & 0xFF;
                gray = (int)( rGray[((argb >> 16) & 0xFF)] + gGray[((argb >> 8) & 0xFF)] + bGray[(argb & 0xFF)]) & 0xFF;
                imageArray[i] = (argb & 0xFF000000) | (gray << 16) | (gray <<  8) | gray;
                i++;
            }   
            
            i += step; 
        }
        
        // Border north
        for (i=northStart; i < imageLength; i++) {
            argb = imageArray[i];
//            gray = (int)(((argb >> 16) & 0xFF)*0.212671f + ((argb >> 8) & 0xFF)*0.715516f + (argb & 0xFF)*0.071169f) & 0xFF;
            gray = (int)( rGray[((argb >> 16) & 0xFF)] + gGray[((argb >> 8) & 0xFF)] + bGray[(argb & 0xFF)]) & 0xFF;
            imageArray[i] = (argb & 0xFF000000) | (gray << 16) | (gray <<  8) | gray;
        }
    }

    /**
     * Flips and paints the result of the 3D rendering.
     *
     * @param g {@inheritDoc}
     */
    @Override
    public void paintComponent(Graphics g) {
        
        super.paintComponent(g); // paint background
        
        if (!isReadyForDrawing()) {
            return;
        }
                        
        // Draw & flip paintImage
        g.drawImage(paintImage,
                // destination: Graphics g, flip lowerY and upperY 
                // dx1    dy1           dx2      dy2
                    0, imageHeight, imageWidth, 0,    
                // source: BufferedImage paintImage
                // sx1    sy1           sx2      sy2
                    0, 0,           imageWidth, imageHeight, null);
 
        // Add further 2D paintings: Text, Shapes, Lines etc.
        
        // Frames per second

        frameCounter++;

        if (frameCounter >= elapsedFrames) {

            endTimePaint = System.nanoTime()/1000000;
            frameDuration = (endTimePaint-startTimePaint)/frameCounter;
            startTimePaint = endTimePaint;

            int fpsPaint = (int)(1000 / frameDuration);

            this.universe.updateFPS(fpsPaint);
            
            // Reset
            frameCounter = 0;
            elapsedFrames = (int)Math.max(1, ((fpsPaint + 0.5) / updatesPerSecond));
        }           
    }
}
